# -*- coding: utf-8 -*-
require File.expand_path('../executor', File.dirname(__FILE__))

describe SPARQLExecutor, "は" do
  include GalaxyTool::Matcher

  before do
    @tool = SPARQLExecutor.new
  end

  it "SPARQLエンドポイント指定として '-e' オプションを受けつける" do
    @tool.should accept_option_with_argument(:endpoint, '-e')
  end
  it "SPARQLエンドポイント指定として '--endpoint' オプションを受けつける" do
    @tool.should accept_option_with_argument(:endpoint, '--endpoint')
  end
  it "'--endpoint' オプションを、前後空白を無視して受けつける" do
    @tool.should evaluate_option('--endpoint', " http://example.com/endpoint\n", "http://example.com/endpoint")
  end
  it "クエリ指定は必須である" do
    @tool.should require_option(:endpoint)
  end

  it "クエリ指定として '-q' オプションを受けつける" do
    @tool.should accept_option_with_argument(:query, '-q')
  end
  it "クエリ指定として '--query' オプションを受けつける" do
    @tool.should accept_option_with_argument(:query, '--query')
  end
  it "クエリ指定として '--query-file' オプションを受けつける" do
    @tool.should accept_option_with_argument(:query, '--query-file')
  end
  it "'--query-file' オプションを、クエリが書かれたファイルのパスとして受けつける" do
    @tool.should evaluate_option('--query-file', stub_file("p53\n").path, 'p53')
  end
  it "クエリ指定は必須である" do
    @tool.should require_option(:query)
  end

  it "出力先指定として '-o' オプションを受けつける" do
    @tool.should accept_option_with_argument(:output, '-o')
  end
  it "出力先指定として '--output' オプションを受けつける" do
    @tool.should accept_option_with_argument(:output, '--output')
  end
  it "出力先指定は必須である" do
    @tool.should require_option(:output)
  end

  it "Info出力先指定として '--info' オプションを受けつける" do
    @tool.should accept_option_with_argument(:info, '--info')
  end
  it "Info出力先の規定値は標準出力である" do
    @tool.should set_default_option_value(:info, '/dev/stdout')
  end

  describe "エンドポイント、クエリ、出力先を指定して実行すると" do
    before do
      @tool.should_receive(:search).and_return do |client, query|
        client.should_receive(:parsed_xml) do |parse|
          REXML::Document.new(<<-EOT).root
<?xml version="1.0" ?>
<sparql xmlns="http://www.w3.org/2005/sparql-results#" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="http://www.w3.org/2001/sw/DataAccess/rf1/result2.xsd">
 <head>
  <variable name="s"/>
  <variable name="p"/>
  <variable name="o"/>
 </head>
 <results distinct="false" ordered="true">
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-uri-fn</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl</uri></binding>
   <binding name="o"><literal>  ^{custom-verbatim-1}^_INVERSE (__id2in (^{tree}^))</literal></binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-uri-fn</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl</uri></binding>
   <binding name="o"><literal>  ^{custom-verbatim-1}^_INVERSE (^{tree}^)</literal></binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl</uri></binding>
   <binding name="o"><literal>  ^{custom-verbatim-1}^_INVERSE (^{tree}^)</literal></binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl</uri></binding>
   <binding name="o"><literal>  ^{custom-verbatim-1}^_INVERSE (__ro2sq /*l*/ (^{tree}^))</literal></binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-dt</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfSqlvalTmpl</uri></binding>
   <binding name="o"><literal>  ^{sqlval-of-tree}^^{as-name-0}^^{comma-cut}^NULL^{as-name-1}^</literal></binding>
  </result>
 </results>
</sparql>
          EOT
        end

        client.query(query)
      end

      @url = "http://www.semantic-systems-biology.org/biogateway/endpoint"
      @query = "select * where { ?s ?p ?o . } limit 5"
      @info = stub_stdout
      @output = stub_file
      @tool.run ["-q", @query, "-e", @url, "-o", @output.path]
    end

    it "SPARQL検索を行い、ID、URL、タイトル、ランクをタブ区切りで表示する" do
      @output.should be_output(<<-EOT)
?s	?p	?o
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-uri-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl>	"  ^{custom-verbatim-1}^_INVERSE (__id2in (^{tree}^))"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-uri-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl>	"  ^{custom-verbatim-1}^_INVERSE (^{tree}^)"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl>	"  ^{custom-verbatim-1}^_INVERSE (^{tree}^)"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl>	"  ^{custom-verbatim-1}^_INVERSE (__ro2sq /*l*/ (^{tree}^))"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-dt>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfSqlvalTmpl>	"  ^{sqlval-of-tree}^^{as-name-0}^^{comma-cut}^NULL^{as-name-1}^"
      EOT
    end
  end  

  describe "結果の途中に改行のテキストノードを含むレスポンスを得るようなエンドポイントに対するクエリだったとしても" do
    before do
      @tool.should_receive(:search).and_return do |client, query|
        client.should_receive(:parsed_xml) do |parse|
          REXML::Document.new(<<-EOT).root
<?xml version="1.0" ?>
<sparql xmlns="http://www.w3.org/2005/sparql-results#" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="http://www.w3.org/2001/sw/DataAccess/rf1/result2.xsd">
 <head>
  <variable name="s"/>
  <variable name="p"/>
  <variable name="o"/>
 </head>
 <results distinct="false" ordered="true">
  <result>
   <binding name="s">
    <uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-uri-fn</uri>
   </binding>
   <binding name="p">
    <uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl</uri>
   </binding>
   <binding name="o">
    <literal>  ^{custom-verbatim-1}^_INVERSE (__id2in (^{tree}^))</literal>
   </binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-uri-fn</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl</uri></binding>
   <binding name="o"><literal>  ^{custom-verbatim-1}^_INVERSE (^{tree}^)</literal></binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl</uri></binding>
   <binding name="o"><literal>  ^{custom-verbatim-1}^_INVERSE (^{tree}^)</literal></binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl</uri></binding>
   <binding name="o"><literal>  ^{custom-verbatim-1}^_INVERSE (__ro2sq /*l*/ (^{tree}^))</literal></binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-dt</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfSqlvalTmpl</uri></binding>
   <binding name="o"><literal>  ^{sqlval-of-tree}^^{as-name-0}^^{comma-cut}^NULL^{as-name-1}^</literal></binding>
  </result>
 </results>
</sparql>
          EOT
        end

        client.query(query)
      end

      @url = "http://www.semantic-systems-biology.org/biogateway/endpoint"
      @query = "select * where { ?s ?p ?o . } limit 5"
      @info = stub_stdout
      @output = stub_file
      @tool.run ["-q", @query, "-e", @url, "-o", @output.path]
    end

    it "SPARQL検索を行い、ID、URL、タイトル、ランクをタブ区切りで表示する" do
      @output.should be_output(<<-EOT)
?s	?p	?o
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-uri-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl>	"  ^{custom-verbatim-1}^_INVERSE (__id2in (^{tree}^))"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-uri-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl>	"  ^{custom-verbatim-1}^_INVERSE (^{tree}^)"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl>	"  ^{custom-verbatim-1}^_INVERSE (^{tree}^)"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl>	"  ^{custom-verbatim-1}^_INVERSE (__ro2sq /*l*/ (^{tree}^))"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-dt>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfSqlvalTmpl>	"  ^{sqlval-of-tree}^^{as-name-0}^^{comma-cut}^NULL^{as-name-1}^"
      EOT
    end
  end

  describe "Blank Nodeを含むレスポンスとなるクエリだったとしても" do
    before do
      @tool.should_receive(:search).and_return do |client, query|
        client.should_receive(:parsed_xml) do |parse|
          REXML::Document.new(<<-EOT).root
<?xml version="1.0" ?>
<sparql xmlns="http://www.w3.org/2005/sparql-results#" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="http://www.w3.org/2001/sw/DataAccess/rf1/result2.xsd">
 <head>
  <variable name="s"/>
  <variable name="p"/>
  <variable name="o"/>
 </head>
 <results distinct="false" ordered="true">
  <result>
   <binding name="s">
    <bnode>nodeID://b196899188</bnode>
   </binding>
   <binding name="p">
    <uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl</uri>
   </binding>
   <binding name="o">
    <literal>  ^{custom-verbatim-1}^_INVERSE (__id2in (^{tree}^))</literal>
   </binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-uri-fn</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl</uri></binding>
   <binding name="o"><literal>  ^{custom-verbatim-1}^_INVERSE (^{tree}^)</literal></binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl</uri></binding>
   <binding name="o"><literal>  ^{custom-verbatim-1}^_INVERSE (^{tree}^)</literal></binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl</uri></binding>
   <binding name="o"><literal>  ^{custom-verbatim-1}^_INVERSE (__ro2sq /*l*/ (^{tree}^))</literal></binding>
  </result>
  <result>
   <binding name="s"><uri>http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-dt</uri></binding>
   <binding name="p"><uri>http://www.openlinksw.com/schemas/virtrdf#qmfShortOfSqlvalTmpl</uri></binding>
   <binding name="o"><literal>  ^{sqlval-of-tree}^^{as-name-0}^^{comma-cut}^NULL^{as-name-1}^</literal></binding>
  </result>
 </results>
</sparql>
          EOT
        end

        client.query(query)
      end

      @url = "http://www.semantic-systems-biology.org/biogateway/endpoint"
      @query = "select * where { ?s ?p ?o . } limit 5"
      @info = stub_stdout
      @output = stub_file
      @tool.run ["-q", @query, "-e", @url, "-o", @output.path]
    end

    it "SPARQL検索を行い、ID、URL、タイトル、ランクをタブ区切りで表示する" do
      @output.should be_output(<<-EOT)
?s	?p	?o
_:nodeID://b196899188	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl>	"  ^{custom-verbatim-1}^_INVERSE (__id2in (^{tree}^))"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-uri-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl>	"  ^{custom-verbatim-1}^_INVERSE (^{tree}^)"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfUriTmpl>	"  ^{custom-verbatim-1}^_INVERSE (^{tree}^)"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-literal-fn>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfLongTmpl>	"  ^{custom-verbatim-1}^_INVERSE (__ro2sq /*l*/ (^{tree}^))"
<http://www.openlinksw.com/virtrdf-data-formats#sql-varchar-dt>	<http://www.openlinksw.com/schemas/virtrdf#qmfShortOfSqlvalTmpl>	"  ^{sqlval-of-tree}^^{as-name-0}^^{comma-cut}^NULL^{as-name-1}^"
      EOT
    end
  end  

end
