#ifndef ALIGNMENT_UTIL_H
#define ALIGNMENT_UTIL_H

#include <string>
#include <vector>

#include <boost/algorithm/string.hpp>

#include "int_util.h"

// macro to check if read mapped
#define READ_UNMAPPED 4

/**
 * Splits alignment into separate parts
 */
std::vector<std::string> split_alignment(std::string &alignment) {
    	std::vector<std::string> parts;

    	boost::trim_if(alignment, boost::is_any_of("\t "));
	// split on tab delimeter
    	boost::split(parts, alignment, boost::is_any_of("\t "), boost::token_compress_on);
	
    	return parts;
}

/**
 * Validates bit flag
 */
bool is_good_flag(const int &bit_flag) {
    	if( (bit_flag & READ_UNMAPPED) > 0) return false;
    	return true;
}

/**
 * Validates rname
 */
bool is_good_rname(const std::string &rname) {
    	return rname.compare("*") != 0;
}

/**
 * Validates pos
 */
bool is_good_pos(const int &pos) {
    	return pos > 0;
}

/**
 * Validates cigar
 */
bool is_good_cigar(const std::string &cigar) {
    	return cigar.compare("*") != 0;
}

/**
 * Validates seq
 */
bool is_good_seq(const std::string &seq) {
    	return seq.compare("*") != 0;
}

/**
 * Validates alignment fields
 */
bool fields_are_good(std::vector<std::string> &parts) {
    	int bit_flag = s_to_i(parts[1]);
    	int pos = s_to_i(parts[3]);

    	std::string rname = parts[2];
    	std::string cigar = parts[5];
    	std::string seq = parts[9];

    	if(!(is_good_flag(bit_flag))) return false;
    	if(!(is_good_pos(pos))) return false;
    	if(!(is_good_rname(rname))) return false;
    	if(!(is_good_cigar(cigar))) return false;
    	if(!(is_good_seq(seq))) return false;

    	return true;
}

/**
 * Stores alignments that pass validity checks
 */
bool is_good_alignment(std::string &alignment) {
    	std::vector<std::string> alignment_parts;

    	alignment_parts = split_alignment(alignment);

    	if(!(fields_are_good(alignment_parts)))
        	return false;
    	return true;
}

#endif /* ALIGNMENT_UTIL_H */
