/*
 * Decompiled with CFR 0.152.
 */
package edu.unc.genomics.io;

import ed.javatools.BufferedRandomAccessFile;
import edu.ucsc.genome.TrackHeader;
import edu.ucsc.genome.TrackHeaderException;
import edu.unc.genomics.io.Contig;
import edu.unc.genomics.io.FixedStepContig;
import edu.unc.genomics.io.VariableStepContig;
import edu.unc.genomics.io.WigFile;
import edu.unc.genomics.io.WigFileException;
import edu.unc.genomics.util.ChecksumUtils;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;
import org.apache.log4j.Logger;
import org.broad.igv.bbfile.WigItem;

public class TextWigFile
extends WigFile {
    private static final long serialVersionUID = -1092879147867842796L;
    public static final String INDEX_EXTENSION = ".wIdx";
    public static final int KEY_GRANULARITY = 10000;
    private static Logger log = Logger.getLogger(TextWigFile.class);
    private BufferedRandomAccessFile raf;
    private TrackHeader header = new TrackHeader("wiggle_0");
    private List<Contig> contigs = new ArrayList<Contig>();
    private Set<String> chromosomes = new HashSet<String>();
    private long checksum;
    private long numBases = 0L;
    private double total = 0.0;
    private double mean = Double.NaN;
    private double stdev = Double.NaN;
    private double min = Double.MAX_VALUE;
    private double max = Double.MIN_VALUE;

    public TextWigFile(Path p) throws IOException, WigFileException {
        super(p);
        this.raf = new BufferedRandomAccessFile(p.toFile(), "r");
        String headerLine = this.raf.readLine();
        if (headerLine.startsWith("track")) {
            try {
                this.header = TrackHeader.parse(headerLine);
            }
            catch (TrackHeaderException e) {
                log.error((Object)("Error parsing UCSC track header in file: " + p.toString()));
                e.printStackTrace();
            }
        }
        this.checksum = ChecksumUtils.adler32(p);
        Path indexFile = p.resolveSibling(p.getFileName() + INDEX_EXTENSION);
        try {
            this.loadIndex(indexFile, true);
        }
        catch (WigFileException | IOException e) {
            Files.deleteIfExists(indexFile);
            this.generateIndex();
            this.saveIndex(indexFile);
        }
    }

    public TextWigFile(Path p, Path index) throws IOException, WigFileException {
        super(p);
        this.raf = new BufferedRandomAccessFile(p.toFile(), "r");
        String headerLine = this.raf.readLine();
        if (headerLine.startsWith("track")) {
            try {
                this.header = TrackHeader.parse(headerLine);
            }
            catch (TrackHeaderException e) {
                System.err.println("Error parsing UCSC track header in file: " + p.toString());
                System.err.println(e.getMessage());
            }
        }
        this.loadIndex(index, false);
    }

    @Override
    public void close() {
        try {
            this.raf.close();
        }
        catch (IOException e) {
            throw new RuntimeException("Error closing TextWigFile");
        }
    }

    @Override
    public Iterator<WigItem> query(String chr, int low, int high) throws IOException, WigFileException {
        if (!this.includes(chr, low, high)) {
            throw new WigFileException("WigFile does not contain data for region: " + chr + ":" + low + "-" + high);
        }
        if (low > high) {
            throw new WigFileException("Query start > stop!");
        }
        List<Contig> relevantContigs = this.getContigsForQuery(chr, low, high);
        return new TextWigIterator(this.raf, relevantContigs.iterator(), chr, low, high);
    }

    private List<Contig> getContigsForQuery(String chr, int low, int high) {
        ArrayList<Contig> relevantContigs = new ArrayList<Contig>();
        for (Contig c : this.contigs) {
            if (!c.getChr().equals(chr) || c.getStop() < low || c.getStart() > high) continue;
            relevantContigs.add(c);
        }
        return relevantContigs;
    }

    @Override
    public String toString() {
        StringBuilder s = new StringBuilder("ASCII Text Wiggle file: " + this.header.toString() + "\n");
        s.append("Chromosomes:\n");
        for (String chr : this.chromosomes) {
            s.append('\t').append(chr).append(" start=").append(this.getChrStart(chr)).append(" stop=").append(this.getChrStop(chr)).append('\n');
        }
        s.append("Contigs:\n");
        for (Contig c : this.contigs) {
            s.append("\t").append(c.toString()).append('\n');
        }
        s.append("Basic Statistics:\n");
        s.append("\tMean:\t\t\t").append(this.mean()).append("\n");
        s.append("\tStandard Deviation:\t").append(this.stdev()).append("\n");
        s.append("\tTotal:\t\t\t").append(this.total()).append("\n");
        s.append("\tBases Covered:\t\t").append(this.numBases()).append("\n");
        s.append("\tMin value:\t\t").append(this.min()).append("\n");
        s.append("\tMax value:\t\t").append(this.max());
        return s.toString();
    }

    @Override
    public Set<String> chromosomes() {
        return this.chromosomes;
    }

    @Override
    public int getChrStart(String chr) {
        if (!this.includes(chr)) {
            return -1;
        }
        int start = Integer.MAX_VALUE;
        for (Contig c : this.contigs) {
            if (!c.getChr().equals(chr) || c.getStart() >= start) continue;
            start = c.getStart();
        }
        return start;
    }

    @Override
    public int getChrStop(String chr) {
        int stop = -1;
        for (Contig c : this.contigs) {
            if (!c.getChr().equals(chr) || c.getStop() <= stop) continue;
            stop = c.getStop();
        }
        return stop;
    }

    @Override
    public boolean includes(String chr, int start, int stop) {
        return this.includes(chr) && this.getChrStart(chr) <= start && this.getChrStop(chr) >= stop;
    }

    @Override
    public boolean includes(String chr) {
        return this.chromosomes.contains(chr);
    }

    @Override
    public long numBases() {
        return this.numBases;
    }

    @Override
    public double total() {
        return this.total;
    }

    @Override
    public double mean() {
        return this.mean;
    }

    @Override
    public double stdev() {
        return this.stdev;
    }

    @Override
    public double min() {
        return this.min;
    }

    @Override
    public double max() {
        return this.max;
    }

    private void generateIndex() throws IOException, WigFileException {
        log.debug((Object)("Indexing ascii text Wig file: " + this.p.getFileName().toString()));
        this.raf.seek(0L);
        long lineNum = 0L;
        String line = this.raf.readLine();
        if (!line.startsWith("track")) {
            this.raf.seek(0L);
        } else {
            ++lineNum;
        }
        this.contigs = new ArrayList<Contig>();
        Contig contig = null;
        int bp = 0;
        double sumOfSquares = 0.0;
        long cursor = this.raf.getFilePointer();
        while ((line = this.raf.readLine()) != null) {
            ++lineNum;
            if (line.startsWith("fixedStep") || line.startsWith("variableStep")) {
                if (this.contigs.size() > 0) {
                    contig.setStopLine(lineNum - 1L);
                    contig.setStop(bp + contig.getSpan() - 1);
                }
                contig = Contig.parse(line);
                this.contigs.add(contig);
                contig.setStartLine(lineNum + 1L);
                if (contig instanceof VariableStepContig) {
                    cursor = this.raf.getFilePointer();
                    String firstLine = this.raf.readLine();
                    int delim = firstLine.indexOf(9);
                    if (delim == -1) {
                        throw new WigFileException("Illegal format in variableStep contig, line " + lineNum);
                    }
                    try {
                        bp = Integer.parseInt(firstLine.substring(0, delim));
                    }
                    catch (NumberFormatException e) {
                        throw new WigFileException("Illegal format in variableStep contig, line " + lineNum);
                    }
                    contig.setStart(bp);
                    this.raf.seek(cursor);
                } else {
                    bp = contig.getStart() - ((FixedStepContig)contig).getStep();
                }
            } else {
                double value;
                if (contig instanceof FixedStepContig) {
                    bp += ((FixedStepContig)contig).getStep();
                    try {
                        value = Double.parseDouble(line);
                    }
                    catch (NumberFormatException e) {
                        throw new WigFileException("Illegal format in fixedStep contig, line " + lineNum);
                    }
                }
                int delim = line.indexOf(9);
                if (delim == -1) {
                    throw new WigFileException("Illegal format in variableStep contig, line " + lineNum);
                }
                try {
                    bp = Integer.parseInt(line.substring(0, delim));
                    value = Double.parseDouble(line.substring(delim + 1));
                }
                catch (NumberFormatException e) {
                    throw new WigFileException("Illegal format in variableStep contig, line " + lineNum);
                }
                if (value < this.min) {
                    this.min = value;
                }
                if (value > this.max) {
                    this.max = value;
                }
                this.numBases += (long)contig.getSpan();
                this.total += (double)contig.getSpan() * value;
                sumOfSquares += (double)contig.getSpan() * value * value;
                if ((lineNum - contig.getStartLine()) % 10000L == 0L) {
                    contig.storeIndex(bp, cursor);
                }
            }
            if ((lineNum + 1L - contig.getStartLine()) % 10000L != 0L) continue;
            cursor = this.raf.getFilePointer();
        }
        if (this.contigs.size() > 0) {
            contig.setStopLine(lineNum);
            contig.setStop(bp + contig.getSpan() - 1);
        }
        this.mean = this.total / (double)this.numBases;
        double variance = (sumOfSquares - this.total * this.mean) / (double)this.numBases;
        this.stdev = Math.sqrt(variance);
        this.chromosomes = new HashSet<String>();
        for (Contig c : this.contigs) {
            this.chromosomes.add(c.getChr());
        }
    }

    private void loadIndex(Path p, boolean matchChecksum) throws IOException, WigFileException {
        log.debug((Object)"Attempting to load Wig file index from disk");
        try (InputStream is = Files.newInputStream(p, new OpenOption[0]);){
            BufferedInputStream bis = new BufferedInputStream(is);
            ObjectInputStream dis = new ObjectInputStream(bis);
            long version = dis.readLong();
            if (version != -1092879147867842796L) {
                throw new WigFileException("Cannot load index from older version!");
            }
            long indexChecksum = dis.readLong();
            if (matchChecksum && indexChecksum != this.checksum) {
                throw new WigFileException("Index does not match checksum of Wig file!");
            }
            this.numBases = dis.readLong();
            this.total = dis.readDouble();
            this.mean = dis.readDouble();
            this.stdev = dis.readDouble();
            this.min = dis.readDouble();
            this.max = dis.readDouble();
            try {
                int numChromosomes = dis.readInt();
                this.chromosomes = new HashSet<String>(numChromosomes);
                for (int i = 0; i < numChromosomes; ++i) {
                    String chr = (String)dis.readObject();
                    this.chromosomes.add(chr);
                }
                int numContigs = dis.readInt();
                this.contigs = new ArrayList<Contig>(numContigs);
                for (int i = 0; i < numContigs; ++i) {
                    Contig contig = (Contig)dis.readObject();
                    this.contigs.add(contig);
                }
                dis.close();
                bis.close();
            }
            catch (ClassNotFoundException e) {
                log.error((Object)"ClassNotFoundException while loading Wig index from file");
                e.printStackTrace();
                throw new WigFileException("ClassNotFoundException while trying to load Wig index from file");
            }
        }
    }

    private void saveIndex(Path p) throws IOException {
        log.debug((Object)"Writing Wig index information to disk");
        try (OutputStream os = Files.newOutputStream(p, new OpenOption[0]);){
            BufferedOutputStream bos = new BufferedOutputStream(os);
            ObjectOutputStream dos = new ObjectOutputStream(bos);
            dos.writeLong(-1092879147867842796L);
            dos.writeLong(this.checksum);
            dos.writeLong(this.numBases);
            dos.writeDouble(this.total);
            dos.writeDouble(this.mean);
            dos.writeDouble(this.stdev);
            dos.writeDouble(this.min);
            dos.writeDouble(this.max);
            dos.writeInt(this.chromosomes.size());
            for (String chr : this.chromosomes) {
                dos.writeObject(chr);
            }
            dos.writeInt(this.contigs.size());
            for (Contig c : this.contigs) {
                dos.writeObject(c);
            }
            dos.close();
            bos.close();
        }
        catch (IOException e) {
            log.error((Object)("Error saving Wig index information to disk!: " + e.getMessage()));
            e.printStackTrace();
            Files.deleteIfExists(p);
        }
    }

    private static class TextWigIterator
    implements Iterator<WigItem> {
        private final RandomAccessFile raf;
        private final String chr;
        private final int start;
        private final int stop;
        private final Iterator<Contig> relevantContigsIter;
        private Iterator<WigItem> currentContigIter;

        public TextWigIterator(RandomAccessFile raf, Iterator<Contig> relevantContigsIter, String chr, int start, int stop) {
            this.raf = raf;
            this.chr = chr;
            this.relevantContigsIter = relevantContigsIter;
            this.start = start;
            this.stop = stop;
        }

        @Override
        public boolean hasNext() {
            if (this.currentContigIter == null || !this.currentContigIter.hasNext()) {
                return this.advanceContig();
            }
            return true;
        }

        @Override
        public WigItem next() {
            if (this.hasNext()) {
                return this.currentContigIter.next();
            }
            throw new NoSuchElementException("No more WigItem elements available");
        }

        @Override
        public void remove() throws UnsupportedOperationException {
            throw new UnsupportedOperationException("Cannot remove records from Wig file");
        }

        private boolean advanceContig() {
            while (this.relevantContigsIter.hasNext()) {
                Contig currentContig = this.relevantContigsIter.next();
                try {
                    this.currentContigIter = currentContig.query(this.raf, this.chr, this.start, this.stop);
                    if (!this.currentContigIter.hasNext()) continue;
                    return true;
                }
                catch (WigFileException | IOException e) {
                    log.error((Object)("Error querying Contig: " + currentContig.toString()));
                    e.printStackTrace();
                    throw new RuntimeException("Error querying Contig: " + currentContig.toString());
                }
            }
            return false;
        }
    }
}

