/*
 * Decompiled with CFR 0.152.
 */
package net.sf.samtools;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import net.sf.samtools.AbstractBAMFileIndex;
import net.sf.samtools.BAMFileSpan;
import net.sf.samtools.BAMIndexContent;
import net.sf.samtools.BAMIndexMetaData;
import net.sf.samtools.BAMIndexWriter;
import net.sf.samtools.Bin;
import net.sf.samtools.BinaryBAMIndexWriter;
import net.sf.samtools.CachingBAMFileIndex;
import net.sf.samtools.Chunk;
import net.sf.samtools.LinearIndex;
import net.sf.samtools.SAMException;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileSource;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMSequenceRecord;
import net.sf.samtools.TextualBAMIndexWriter;
import net.sf.samtools.util.BlockCompressedFilePointerUtil;

public class BAMIndexer {
    private final int numReferences;
    private final BAMIndexWriter outputWriter;
    private int currentReference = 0;
    private final BAMIndexBuilder indexBuilder;

    public BAMIndexer(File output, SAMFileHeader fileHeader) {
        this.numReferences = fileHeader.getSequenceDictionary().size();
        this.indexBuilder = new BAMIndexBuilder(fileHeader);
        this.outputWriter = new BinaryBAMIndexWriter(this.numReferences, output);
    }

    public void processAlignment(SAMRecord rec) {
        try {
            int reference = rec.getReferenceIndex();
            if (reference != -1 && reference != this.currentReference) {
                this.advanceToReference(reference);
            }
            this.indexBuilder.processAlignment(rec);
        }
        catch (Exception e) {
            throw new SAMException("Exception creating BAM index for record " + rec, e);
        }
    }

    public void finish() {
        this.advanceToReference(this.numReferences);
        this.outputWriter.writeNoCoordinateRecordCount(this.indexBuilder.getNoCoordinateRecordCount());
        this.outputWriter.close();
    }

    private void advanceToReference(int nextReference) {
        while (this.currentReference < nextReference) {
            BAMIndexContent content = this.indexBuilder.processReference(this.currentReference);
            this.outputWriter.writeReference(content);
            ++this.currentReference;
            this.indexBuilder.startNewReference();
        }
    }

    public static void createAndWriteIndex(File input, File output, boolean textOutput) {
        CachingBAMFileIndex existingIndex = new CachingBAMFileIndex(input, null);
        int n_ref = existingIndex.getNumberOfReferences();
        BAMIndexWriter outputWriter = textOutput ? new TextualBAMIndexWriter(n_ref, output) : new BinaryBAMIndexWriter(n_ref, output);
        try {
            for (int i = 0; i < n_ref; ++i) {
                outputWriter.writeReference(existingIndex.getQueryResults(i));
            }
            outputWriter.writeNoCoordinateRecordCount(existingIndex.getNoCoordinateCount());
            outputWriter.close();
        }
        catch (Exception e) {
            throw new SAMException("Exception creating BAM index", e);
        }
    }

    private class BAMIndexBuilder {
        private final SAMFileHeader bamHeader;
        private Bin[] bins;
        private int binsSeen = 0;
        private final long[] index = new long[LinearIndex.MAX_LINEAR_INDEX_SIZE];
        private int largestIndexSeen = -1;
        private BAMIndexMetaData indexStats = new BAMIndexMetaData();

        BAMIndexBuilder(SAMFileHeader header) {
            this.bamHeader = header;
        }

        public void processAlignment(SAMRecord rec) {
            Bin bin;
            int binNum;
            this.indexStats.recordMetaData(rec);
            int alignmentStart = rec.getAlignmentStart();
            if (alignmentStart == 0) {
                return;
            }
            int reference = rec.getReferenceIndex();
            if (reference != BAMIndexer.this.currentReference) {
                throw new SAMException("Unexpected reference " + reference + " when constructing index for " + BAMIndexer.this.currentReference + " for record " + rec);
            }
            Integer binNumber = rec.getIndexingBin();
            int n = binNum = binNumber == null ? rec.computeIndexingBin() : binNumber.intValue();
            if (this.bins == null) {
                SAMSequenceRecord seq = this.bamHeader.getSequence(reference);
                this.bins = seq == null ? new Bin[37451] : new Bin[AbstractBAMFileIndex.getMaxBinNumberForSequenceLength(seq.getSequenceLength()) + 1];
            }
            if (this.bins[binNum] != null) {
                bin = this.bins[binNum];
            } else {
                this.bins[binNum] = bin = new Bin(reference, binNum);
                ++this.binsSeen;
            }
            SAMFileSource source = rec.getFileSource();
            if (source == null) {
                throw new SAMException("No source (virtual file offsets); needed for indexing on BAM Record " + rec);
            }
            Chunk newChunk = ((BAMFileSpan)source.getFilePointer()).getSingleChunk();
            long chunkStart = newChunk.getChunkStart();
            long chunkEnd = newChunk.getChunkEnd();
            List<Chunk> oldChunks = bin.getChunkList();
            if (!bin.containsChunks()) {
                bin.addInitialChunk(newChunk);
            } else {
                Chunk lastChunk = bin.getLastChunk();
                if (BlockCompressedFilePointerUtil.areInSameOrAdjacentBlocks(lastChunk.getChunkEnd(), chunkStart)) {
                    lastChunk.setChunkEnd(chunkEnd);
                } else {
                    oldChunks.add(newChunk);
                    bin.setLastChunk(newChunk);
                }
            }
            int alignmentEnd = rec.getAlignmentEnd();
            int startWindow = LinearIndex.convertToLinearIndexOffset(alignmentStart);
            int endWindow = alignmentEnd == 0 ? (startWindow = LinearIndex.convertToLinearIndexOffset(alignmentStart - 1)) : LinearIndex.convertToLinearIndexOffset(alignmentEnd);
            if (endWindow > this.largestIndexSeen) {
                this.largestIndexSeen = endWindow;
            }
            for (int win = startWindow; win <= endWindow; ++win) {
                if (this.index[win] != 0L && chunkStart >= this.index[win]) continue;
                this.index[win] = chunkStart;
            }
        }

        public BAMIndexContent processReference(int reference) {
            if (reference != BAMIndexer.this.currentReference) {
                throw new SAMException("Unexpected reference " + reference + " when constructing index for " + BAMIndexer.this.currentReference);
            }
            if (this.binsSeen == 0) {
                return null;
            }
            long[] newIndex = new long[this.largestIndexSeen + 1];
            long lastNonZeroOffset = 0L;
            for (int i = 0; i <= this.largestIndexSeen; ++i) {
                if (this.index[i] == 0L) {
                    this.index[i] = lastNonZeroOffset;
                } else {
                    lastNonZeroOffset = this.index[i];
                }
                newIndex[i] = this.index[i];
            }
            LinearIndex linearIndex = new LinearIndex(reference, 0, newIndex);
            return new BAMIndexContent(reference, this.bins, this.binsSeen, this.indexStats, linearIndex);
        }

        public long getNoCoordinateRecordCount() {
            return this.indexStats.getNoCoordinateRecordCount();
        }

        void startNewReference() {
            this.bins = null;
            if (this.binsSeen > 0) {
                Arrays.fill(this.index, 0L);
            }
            this.binsSeen = 0;
            this.largestIndexSeen = -1;
            this.indexStats.newReference();
        }
    }
}

