/*
 * Decompiled with CFR 0.152.
 */
package edu.unc.genomics.io;

import ed.javatools.BufferedRandomAccessFile;
import edu.ucsc.genome.TrackHeader;
import edu.ucsc.genome.TrackHeaderException;
import edu.unc.genomics.Contig;
import edu.unc.genomics.Interval;
import edu.unc.genomics.io.ContigIndex;
import edu.unc.genomics.io.FixedStepContigIndex;
import edu.unc.genomics.io.WigFileException;
import edu.unc.genomics.io.WigFileFormatException;
import edu.unc.genomics.io.WigFileReader;
import edu.unc.genomics.util.ChecksumUtils;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.math3.stat.descriptive.SummaryStatistics;
import org.apache.log4j.Logger;

public class TextWigFileReader
extends WigFileReader {
    private static final long serialVersionUID = 7L;
    public static final String INDEX_EXTENSION = ".idx";
    public static final int KEY_GRANULARITY = 10000;
    private static Logger log = Logger.getLogger(TextWigFileReader.class);
    private BufferedRandomAccessFile raf;
    private Map<String, List<ContigIndex>> contigs = new HashMap<String, List<ContigIndex>>();
    private long checksum;
    private SummaryStatistics stats;

    public TextWigFileReader(Path p) throws IOException, WigFileFormatException {
        super(p);
        log.debug((Object)("Opening ASCII-text Wig file " + p));
        this.raf = new BufferedRandomAccessFile(p.toFile(), "r");
        String headerLine = this.raf.readLine2();
        if (headerLine.startsWith("track")) {
            try {
                this.header = TrackHeader.parse(headerLine);
            }
            catch (TrackHeaderException e) {
                log.error((Object)("Error parsing UCSC track header in file: " + p));
                e.printStackTrace();
            }
        }
        this.checksum = ChecksumUtils.crc32(p);
        Path indexFile = p.resolveSibling(p.getFileName() + INDEX_EXTENSION);
        try {
            this.loadIndex(indexFile, true);
        }
        catch (WigFileException | IOException e) {
            Files.deleteIfExists(indexFile);
            this.generateIndex();
            this.saveIndex(indexFile);
        }
    }

    public TextWigFileReader(Path p, Path index) throws IOException, WigFileException {
        super(p);
        log.debug((Object)("Opening ASCII-text Wig file " + p + " with index " + index));
        this.raf = new BufferedRandomAccessFile(p.toFile(), "r");
        String headerLine = this.raf.readLine2();
        if (headerLine.startsWith("track")) {
            try {
                this.header = TrackHeader.parse(headerLine);
            }
            catch (TrackHeaderException e) {
                System.err.println("Error parsing UCSC track header in file: " + p.toString());
                System.err.println(e.getMessage());
            }
        }
        this.loadIndex(index, false);
    }

    @Override
    public void close() {
        log.debug((Object)("Closing Wig file reader " + this.p));
        try {
            this.raf.close();
        }
        catch (IOException e) {
            throw new RuntimeException("Error closing TextWigFile");
        }
    }

    @Override
    public Contig query(Interval interval) throws IOException, WigFileException {
        if (!this.includes(interval)) {
            throw new WigFileException("WigFile does not contain data for region: " + interval);
        }
        float[] values = new float[interval.length()];
        Arrays.fill(values, Float.NaN);
        for (ContigIndex c : this.getContigsOverlappingInterval(interval)) {
            c.fill(this.raf, interval, values);
        }
        if (interval.isCrick()) {
            ArrayUtils.reverse((float[])values);
        }
        return new Contig(interval, values);
    }

    @Override
    public SummaryStatistics queryStats(Interval interval) throws IOException, WigFileException {
        if (!this.includes(interval)) {
            throw new WigFileException("WigFile does not contain data for region: " + interval);
        }
        SummaryStatistics stats = new SummaryStatistics();
        for (ContigIndex c : this.getContigsOverlappingInterval(interval)) {
            c.fillStats(this.raf, interval, stats);
        }
        return stats;
    }

    private List<ContigIndex> getContigsOverlappingInterval(Interval interval) {
        ArrayList<ContigIndex> relevantContigs = new ArrayList<ContigIndex>();
        for (ContigIndex c : this.contigs.get(interval.getChr())) {
            if (c.getStop() < interval.low() || c.getStart() > interval.high()) continue;
            relevantContigs.add(c);
        }
        log.debug((Object)("Found " + relevantContigs.size() + " contigs overlapping query interval " + interval));
        return relevantContigs;
    }

    @Override
    public String toString() {
        StringBuilder s = new StringBuilder("ASCII Text Wiggle file: " + this.header.toString() + "\n");
        s.append("Chromosomes:\n");
        for (String string : this.chromosomes()) {
            s.append('\t').append(string).append(" start=").append(this.getChrStart(string)).append(" stop=").append(this.getChrStop(string)).append('\n');
        }
        s.append("Contigs:\n");
        for (List list : this.contigs.values()) {
            for (ContigIndex c : list) {
                s.append("\t").append(c.toOutput()).append('\n');
            }
        }
        s.append("Basic Statistics:\n");
        s.append("\tMean:\t\t\t").append(this.mean()).append("\n");
        s.append("\tStandard Deviation:\t").append(this.stdev()).append("\n");
        s.append("\tTotal:\t\t\t").append(this.total()).append("\n");
        s.append("\tBases Covered:\t\t").append(this.numBases()).append("\n");
        s.append("\tMin value:\t\t").append(this.min()).append("\n");
        s.append("\tMax value:\t\t").append(this.max());
        return s.toString();
    }

    @Override
    public Set<String> chromosomes() {
        return this.contigs.keySet();
    }

    @Override
    public int getChrStart(String chr) {
        if (!this.includes(chr)) {
            return -1;
        }
        int start = Integer.MAX_VALUE;
        for (ContigIndex c : this.contigs.get(chr)) {
            if (c.getStart() >= start) continue;
            start = c.getStart();
        }
        return start;
    }

    @Override
    public int getChrStop(String chr) {
        if (!this.includes(chr)) {
            return -1;
        }
        int stop = -1;
        for (ContigIndex c : this.contigs.get(chr)) {
            if (c.getStop() <= stop) continue;
            stop = c.getStop();
        }
        return stop;
    }

    @Override
    public int getChrStep(String chr) {
        if (!this.includes(chr)) {
            return -1;
        }
        int step = Integer.MAX_VALUE;
        for (ContigIndex c : this.contigs.get(chr)) {
            if (c.isVariableStep()) {
                return 1;
            }
            if (((FixedStepContigIndex)c).getStep() >= step) continue;
            step = ((FixedStepContigIndex)c).getStep();
        }
        return step;
    }

    @Override
    public int getChrSpan(String chr) {
        if (!this.includes(chr)) {
            return -1;
        }
        int span = Integer.MAX_VALUE;
        for (ContigIndex c : this.contigs.get(chr)) {
            if (c.getSpan() >= span) continue;
            span = c.getSpan();
        }
        return span;
    }

    @Override
    public boolean includes(String chr, int start, int stop) {
        return this.includes(chr) && this.getChrStart(chr) <= start && this.getChrStop(chr) >= stop;
    }

    @Override
    public boolean includes(String chr) {
        return this.contigs.containsKey(chr);
    }

    @Override
    public long numBases() {
        return this.stats.getN();
    }

    @Override
    public double total() {
        return this.stats.getSum();
    }

    @Override
    public double mean() {
        return this.stats.getMean();
    }

    @Override
    public double stdev() {
        return Math.sqrt(this.stats.getPopulationVariance());
    }

    @Override
    public double min() {
        return this.stats.getMin();
    }

    @Override
    public double max() {
        return this.stats.getMax();
    }

    private void generateIndex() throws IOException, WigFileFormatException {
        log.debug((Object)("Indexing ASCII text Wig file: " + this.p));
        this.raf.seek(0L);
        long lineNum = 0L;
        String line = this.raf.readLine2();
        if (!line.startsWith("track")) {
            this.raf.seek(0L);
        } else {
            ++lineNum;
        }
        this.stats = new SummaryStatistics();
        this.contigs = new HashMap<String, List<ContigIndex>>();
        ContigIndex contig = null;
        int bp = 0;
        long cursor = this.raf.getFilePointer();
        int count = 0;
        while ((line = this.raf.readLine2()) != null) {
            ++lineNum;
            if (line.startsWith(Contig.Type.FIXEDSTEP.getId()) || line.startsWith(Contig.Type.VARIABLESTEP.getId())) {
                if (this.contigs.size() > 0) {
                    contig.setStopLine(lineNum - 1L);
                    contig.setStop(bp + contig.getSpan() - 1);
                }
                contig = ContigIndex.parseHeader(line);
                log.debug((Object)("Found contig header: " + line + " (line " + lineNum + ")"));
                if (!this.contigs.containsKey(contig.getChr())) {
                    this.contigs.put(contig.getChr(), new ArrayList());
                }
                this.contigs.get(contig.getChr()).add(contig);
                contig.setStartLine(lineNum + 1L);
                if (contig.isVariableStep()) {
                    cursor = this.raf.getFilePointer();
                    String firstLine = this.raf.readLine2();
                    int delim = firstLine.indexOf(9);
                    if (delim == -1) {
                        throw new WigFileFormatException("Illegal format in variableStep contig, line " + lineNum);
                    }
                    try {
                        bp = Integer.parseInt(firstLine.substring(0, delim));
                    }
                    catch (NumberFormatException e) {
                        throw new WigFileFormatException("Illegal format in variableStep contig, line " + lineNum);
                    }
                    contig.setStart(bp);
                    this.raf.seek(cursor);
                } else {
                    bp = contig.getStart() - ((FixedStepContigIndex)contig).getStep();
                }
            } else {
                double value;
                if (contig.isFixedStep()) {
                    bp += ((FixedStepContigIndex)contig).getStep();
                    try {
                        value = Double.parseDouble(line);
                    }
                    catch (NumberFormatException e) {
                        throw new WigFileFormatException("Illegal format in fixedStep contig, line " + lineNum);
                    }
                }
                int delim = line.indexOf(9);
                if (delim == -1) {
                    throw new WigFileFormatException("Illegal format in variableStep contig, line " + lineNum);
                }
                try {
                    bp = Integer.parseInt(line.substring(0, delim));
                    value = Double.parseDouble(line.substring(delim + 1));
                }
                catch (NumberFormatException e) {
                    throw new WigFileFormatException("Illegal format in variableStep contig, line " + lineNum);
                }
                if (!Double.isNaN(value) && !Double.isInfinite(value)) {
                    ++count;
                    for (int i = 0; i < contig.getSpan(); ++i) {
                        this.stats.addValue(value);
                    }
                }
                if ((lineNum - contig.getStartLine()) % 10000L == 0L) {
                    contig.storeIndex(bp, cursor);
                }
            }
            if ((lineNum + 1L - contig.getStartLine()) % 10000L != 0L) continue;
            cursor = this.raf.getFilePointer();
        }
        if (contig != null) {
            contig.setStopLine(lineNum);
            contig.setStop(bp + contig.getSpan() - 1);
        }
        log.debug((Object)("Indexed " + count + " entries in Wig file"));
    }

    private void loadIndex(Path p, boolean matchChecksum) throws IOException, WigFileException {
        log.debug((Object)"Attempting to load Wig file index from disk");
        try (ObjectInputStream dis = new ObjectInputStream(new BufferedInputStream(Files.newInputStream(p, new OpenOption[0])));){
            long version = dis.readLong();
            if (version != 7L) {
                log.warn((Object)"Version of index does not match version of Wig file!");
                throw new WigFileException("Cannot load index from older version!");
            }
            long indexChecksum = dis.readLong();
            if (matchChecksum && indexChecksum != this.checksum) {
                log.warn((Object)"Index does not match checksum of Wig file!");
                throw new WigFileException("Index does not match checksum of Wig file!");
            }
            try {
                this.stats = (SummaryStatistics)dis.readObject();
            }
            catch (ClassNotFoundException e) {
                log.error((Object)"ClassNotFoundException while loading Wig statistics from index file");
                e.printStackTrace();
                throw new WigFileException("ClassNotFoundException while trying to load Wig statistics from index file");
            }
            try {
                int numContigs = dis.readInt();
                this.contigs = new HashMap<String, List<ContigIndex>>();
                for (int i = 0; i < numContigs; ++i) {
                    ContigIndex contig = (ContigIndex)dis.readObject();
                    if (!this.contigs.containsKey(contig.getChr())) {
                        this.contigs.put(contig.getChr(), new ArrayList());
                    }
                    this.contigs.get(contig.getChr()).add(contig);
                }
                log.debug((Object)("Loaded index information for " + this.contigs.size() + " contigs"));
            }
            catch (ClassNotFoundException e) {
                log.error((Object)"ClassNotFoundException while loading Wig index from file");
                e.printStackTrace();
                throw new WigFileException("ClassNotFoundException while trying to load Wig index from file");
            }
        }
    }

    private void saveIndex(Path p) throws IOException {
        log.debug((Object)"Writing Wig index information to disk");
        try (ObjectOutputStream dos = new ObjectOutputStream(new BufferedOutputStream(Files.newOutputStream(p, new OpenOption[0])));){
            dos.writeLong(7L);
            dos.writeLong(this.checksum);
            dos.writeObject(this.stats);
            int numContigs = 0;
            for (List<ContigIndex> chromContigs : this.contigs.values()) {
                numContigs += chromContigs.size();
            }
            dos.writeInt(numContigs);
            for (List<ContigIndex> chromContigs : this.contigs.values()) {
                for (ContigIndex c : chromContigs) {
                    dos.writeObject(c);
                }
            }
        }
        catch (IOException e) {
            log.error((Object)("Error saving Wig index information to disk!: " + e.getMessage()));
            e.printStackTrace();
            Files.deleteIfExists(p);
        }
    }
}

