#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import sys
from SMART.Java.Python.structure.TranscriptList import TranscriptList
from SMART.Java.Python.misc.Progress import Progress
from SMART.Java.Python.misc.UnlimitedProgress import UnlimitedProgress

class TranscriptListParser(object):
    """A (quite generic) class that reads a list of transcripts"""

    def __init__(self, fileName, verbosity = 0):
        self.verbosity         = verbosity
        self.fileName          = fileName
        self.nbTranscripts     = None
        self.size              = None
        self.chromosomes       = None
        self.currentTranscript = None
        self.currentLineNb     = 0
        self.previousTranscriptAddress = None
        try:
            self.handle = open(self.fileName)
        except IOError:
            raise Exception("Error! Transcript file '%s' does not exist! Exiting..." % (self.fileName))
        self.skipFirstLines()


    def __del__(self):
        self.close()
        

    def getFileFormats():
        pass
    getFileFormats = staticmethod(getFileFormats)


    def close(self):
        if self.handle != None and not self.handle.close:
            self.handle.close()
        self.handle = None


    def reset(self):
        self.handle.seek(0)
        self.skipFirstLines()
        self.currentTranscript = None
        self.currentLineNb     = 0
        self.currentTranscriptAddress  = self.handle.tell()
        self.currentAddress            = self.handle.tell()


    def gotoAddress(self, address):
        self.reset()
        self.handle.seek(address)
        self.currentTranscriptAddress = address
        self.currentAddress           = address
                
        
    def parse(self):
        transcriptList = TranscriptList()
        progress = Progress(self.getNbTranscripts(), "Reading %s" % (self.fileName), self.verbosity)
        for line in self.handle:
            self.currentLineNb += 1
            transcript = self.parseLine(line)
            transcriptList.addTranscript(transcript)
            progress.inc()
        progress.done()
        return transcriptList


    def getIterator(self):
        self.reset()
        transcript = self.getNextTranscript()
        while transcript != None:
            yield transcript
            transcript = self.getNextTranscript()


    def getCurrentAddress(self):
        return self.currentAddress


    def getCurrentTranscriptAddress(self):
        return self.currentTranscriptAddress


    def getNextTranscript(self):
        self.currentAddress = self.handle.tell()
        line = self.handle.readline()
        while line != "":
            line = line.strip()
            self.currentLineNb += 1
            transcript = self.parseLine(line)
            if transcript != None:
                return transcript
            self.currentAddress = self.handle.tell()
            line = self.handle.readline()
        transcript = self.currentTranscript
        self.currentTranscriptAddress = self.previousTranscriptAddress
        self.currentTranscript = None
        return transcript


    def getInfos(self):
        self.chromosomes = set()
        self.nbTranscripts = 0
        self.size = 0
        self.reset()
        progress = UnlimitedProgress(100000, "Getting information on %s." % (self.fileName), self.verbosity-9)
        transcript = self.getNextTranscript()
        for transcript in self.getIterator():
            self.chromosomes.add(transcript.getChromosome())
            self.nbTranscripts += 1
            self.size += transcript.getSize()
            progress.inc()
        progress.done()
        self.reset()

    
    def getNbTranscripts(self):
        if self.nbTranscripts != None:
            return self.nbTranscripts
        self.getInfos()
        return self.nbTranscripts


    def getNbItems(self):
        return self.getNbTranscripts()


    def getChromosomes(self):
        if self.chromosomes != None:
            return self.chromosomes
        self.getInfos()
        return self.chromosomes
    
    
    def getSize(self):
        if self.size != None:
            return self.size
        self.getInfos()
        return self.size
    
    
    def getNbNucleotides(self):
        return self.getSize()


    def setDefaultTagValue(self, name, value):
        for transcript in self.getIterator():
            transcript.setTag(name, value)
            
    def __eq__(self, o):
        if type(o) is type(self):
            return self.fileName == o.fileName and self.nbTranscripts == o.nbTranscripts and self.size == o.size and self.chromosomes == o.chromosomes
        return False
    
    def __ne__(self, o):
        return not self.__eq__(o)
