#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
from commons.core.writer.TranscriptListWriter import TranscriptListWriter


class WigWriter(TranscriptListWriter):
    """
    A class that writes a transcript list into a file with WIGGLE format
    @ivar fileName: name of the file 
    @type fileName: string
    @ivar handle: handle to the file
    @type handle: file handle
    @ivar header: first lines of the file
    @type header: string
    """


    def __init__(self, fileName, verbosity = 0):
        """
        Constructor
        @param fileName: name of the file 
        @type fileName: string
        @param verbosity: verbosity
        @type verbosity: int
        """
        self.fileName  = fileName
        self.verbosity = verbosity
        self.data      = {-1: {}, 0: {}, 1: {}}
        self.title     = "Reads"
        self.strands   = False
        self.handle    = None


    def __del__(self):
        """
        Destructor
        Actually print the file
        """
        strand2string = {-1: "-", 1: "+", 0: ""}
        self.handle   = open(self.fileName, "w")
        self.handle.write("track type=wiggle_0 name=\"%s\"\n" % (self.title))
        for strand in self.data:
            for chromosome in sorted(self.data[strand]):
                 self.handle.write("variableStep chrom=%s%s\n" % (chromosome, strand2string[strand]))
                 for pos in sorted(self.data[strand][chromosome]):
                     self.handle.write("%d\t%d\n" % (pos, self.data[strand][chromosome][pos]))
        self.handle.close()

        
    @staticmethod
    def getFileFormats():
        """
        Get the format of the file
        """
        return ["wig", "wiggle"]
        
        
    @staticmethod
    def getExtension():
        """
        Get the usual extension for the file
        """
        return "wig"
        
        
    def setTitle(self, title):
        """
        Set the title of the track
        @param title: the title of the track
        @type    title: string
        """
        if title != None:
            self.title = title


    def setStrands(self, strands):
        """
        Consider each strand separately
        @param boolean: whether each strand should be considered separately
        @type  boolean: boolean
        """
        self.strands = strands


    def copyProperties(self, parser):
        """
        Copy the properties collected by a parser, to produce a similar output
        @param bedParser: a parser
        @type    bedParser: class L{TranscriptListWriter<TranscriptListWriter>}
        """
        self.setTitle(parser.title)
        

    def addTranscript(self, transcript):
        """
        Export the given transcript with GBrowse format
        @param transcript: transcript to be printed
        @type transcript: class L{Transcript<Transcript>}
        @return: a string
        """
        chromosome = transcript.getChromosome()
        direction  = transcript.getDirection()
        if not self.strands:
            direction = 0
        if chromosome not in self.data[direction]:
            self.data[direction][chromosome] = {}
        for exon in transcript.getExons():
            for pos in range(exon.getStart(), exon.getEnd()+1):
                if pos not in self.data[direction][chromosome]:
                    self.data[direction][chromosome][pos]  = 1
                else:
                    self.data[direction][chromosome][pos] += 1
