#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2011
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import random, os, time, MySQLdb
from optparse import OptionParser
from commons.core.parsing.ParserChooser import ParserChooser
from commons.core.writer.TranscriptWriter import TranscriptWriter
from SMART.Java.Python.structure.Transcript import Transcript
from SMART.Java.Python.misc.Progress import Progress
from SMART.Java.Python.misc.UnlimitedProgress import UnlimitedProgress


class FindOverlapsWithSeveralIntervalsIndex(object):

    def __init__(self, verbosity):
        self.verbosity = verbosity
        randomNumber   = random.randint(0, 10000)
        self.dbName    = "smartdb"
        if "SMARTTMPPATH" in os.environ:
            self.dbName = os.join(os.environ["SMARTTMPPATH"], self.dbName)
        self.db         = MySQLdb.connect(db = self.dbName)
        self.tableName  = "table_%s" % (randomNumber)
        self.nbQueries  = 0
        self.nbRefs     = 0
        self.nbOverlaps = 0

    def __del__(self):
        cursor = self.db.cursor()
        cursor.execute("DROP TABLE IF EXISTS %s" % (self.tableName))
        

    def setReferenceFile(self, fileName, format):
        cursor = self.db.cursor()
        cursor.execute("CREATE TABLE %s (start INT, end INT)" % (self.tableName))
        cursor.execute("CREATE INDEX index_%s ON %s (start, end)" % (self.tableName, self.tableName))
        chooser = ParserChooser(self.verbosity)
        chooser.findFormat(format)
        parser = chooser.getParser(fileName)
        progress = UnlimitedProgress(1000, "Reading references", self.verbosity)
        for transcript in parser.getIterator():
            start      = transcript.getStart()
            end        = transcript.getEnd()
            cursor     = self.db.cursor()
            cursor.execute("INSERT INTO %s (start, end) VALUES (%d, %d)" % (self.tableName, start, end))
            self.nbRefs += 1
            progress.inc()
        self.db.commit()
        progress.done()

    def setQueryFile(self, fileName, format):
        chooser = ParserChooser(self.verbosity)
        chooser.findFormat(format)
        self.queryParser = chooser.getParser(fileName)
        self.nbQueries = self.queryParser.getNbTranscripts()

    def setOutputFile(self, fileName):
        self.writer = TranscriptWriter(fileName, "gff3", self.verbosity)

    def compare(self):
        progress = Progress(self.nbQueries, "Reading queries", self.verbosity)
        startTime = time.time()
        for queryTranscript in self.queryParser.getIterator():
            queryStart = queryTranscript.getStart()
            queryEnd   = queryTranscript.getEnd()
            command    = "SELECT 1 FROM %s WHERE start <= %d and end >= %d" % (self.tableName, queryEnd, queryStart)
            cursor     = self.db.cursor()
            cursor.execute(command)
            overlap = False
            line = cursor.fetchone()
            while line:
                overlap = True
                line    = cursor.fetchone()
            if overlap:
                self.writer.addTranscript(queryTranscript)
                self.nbOverlaps += 1
            progress.inc()
        progress.done()
        endTime = time.time()
        self.timeSpent = endTime - startTime

    def displayResults(self):
        print "# queries:  %d" % (self.nbQueries)
        print "# refs:     %d" % (self.nbRefs)
        print "# overlaps: %d" % (self.nbOverlaps)
        print "time:       %.2gs" % (self.timeSpent)

    def run(self):
        self.compare()
        self.displayResults()

if __name__ == "__main__":
    
    description = "Find Overlaps With Several Intervals Using Indices v1.0.1: Use MySQL to compare intervals. [Category: Personal]"

    parser = OptionParser(description = description)
    parser.add_option("-i", "--input1",      dest="inputFileName1", action="store",            type="string", help="query input file [compulsory] [format: file in transcript format given by -f]")
    parser.add_option("-f", "--format1",     dest="format1",        action="store",            type="string", help="format of previous file [compulsory] [format: transcript file format]")
    parser.add_option("-j", "--input2",      dest="inputFileName2", action="store",            type="string", help="reference input file [compulsory] [format: file in transcript format given by -g]")
    parser.add_option("-g", "--format2",     dest="format2",        action="store",            type="string", help="format of previous file [compulsory] [format: transcript file format]")
    parser.add_option("-o", "--output",      dest="outputFileName", action="store",            type="string", help="output file [format: output file in GFF3 format]")
    parser.add_option("-v", "--verbosity",   dest="verbosity",      action="store", default=1, type="int",    help="trace level [format: int]")
    (options, args) = parser.parse_args()

    fowsii = FindOverlapsWithSeveralIntervalsIndex(options.verbosity)
    fowsii.setQueryFile(options.inputFileName1, options.format1)
    fowsii.setReferenceFile(options.inputFileName2, options.format2)
    fowsii.setOutputFile(options.outputFileName)
    fowsii.run()


