# Copyright INRA (Institut National de la Recherche Agronomique)
# http://www.inra.fr
# http://urgi.versailles.inra.fr
#
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software.  You can  use, 
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info". 
#
# As a counterpart to the access to the source code and  rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty  and the software's author,  the holder of the
# economic rights,  and the successive licensors  have only  limited
# liability. 
#
# In this respect, the user's attention is drawn to the risks associated
# with loading,  using,  modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean  that it is complicated to manipulate,  and  that  also
# therefore means  that it is reserved for developers  and  experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or 
# data to be ensured and,  more generally, to use and operate it in the 
# same conditions as regards security. 
#
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.


import os
import sys
import unittest
import time
import shutil
from commons.core.utils.FileUtils import FileUtils


class Test_FileUtils( unittest.TestCase ):
    
    def setUp( self ):
        self._uniqId = "%s_%s" % ( time.strftime("%Y%m%d%H%M%S"), os.getpid() )
        
    def tearDown( self ):
        self._uniqId = ""
        
    def test_getNbLinesInSingleFile_non_empty( self ):
        file = "dummyFile_%s" % ( self._uniqId )
        f = open( file, "w" )
        f.write( "line1\n" )
        f.write( "line2\n" )
        f.write( "line3" )
        f.close()
        exp = 3
        obs = FileUtils.getNbLinesInSingleFile( file )
        self.assertEquals( exp, obs )
        os.remove( file )
        
    def test_getNbLinesInSingleFile_non_empty_endEmptyLine( self ):
        file = "dummyFile_%s" % ( self._uniqId )
        f = open( file, "w" )
        f.write( "line1\n" )
        f.write( "line2\n" )
        f.write( "line3\n" )
        f.write( "\n" )
        f.close()
        exp = 3
        obs = FileUtils.getNbLinesInSingleFile( file )
        self.assertEquals( exp, obs )
        os.remove( file )
        
    def test_getNbLinesInSingleFile_empty( self ):
        file = "dummyFile_%s" % ( self._uniqId )
        os.system( "touch %s" % ( file ) )
        exp = 0
        obs = FileUtils.getNbLinesInSingleFile( file )
        self.assertEquals( exp, obs )
        os.remove( file )
        
    def test_getNbLinesInFileList_non_empty( self ):
        f = open("dummy1.txt", "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3")
        f.close()
        f = open("dummy2.txt", "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3")
        f.close()
        f = open("dummy3.txt", "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3")
        f.close()
        lFiles = [ "dummy1.txt", "dummy2.txt", "dummy3.txt" ]
        exp = 9
        obs = FileUtils.getNbLinesInFileList( lFiles )
        self.assertEqual( exp, obs )
        for f in lFiles:
            os.remove( f )
            
    def test_catFilesByPattern( self ):
        f = open("dummy1.txt", "w")
        f.write("line11\n")
        f.write("line12\n")
        f.write("line13")
        f.close()
        f = open("dummy2.txt", "w")
        f.write("line21\n")
        f.write("line22\n")
        f.write("line23\n")
        f.close()
        f = open("dummy3.txt", "w")
        f.write("line31\n")
        f.write("line32\n")
        f.write("line33")
        f.close()
        lFiles = [ "dummy1.txt", "dummy2.txt", "dummy3.txt" ]
        outFile = "concatFiles.txt"
        FileUtils.catFilesByPattern( "dummy*.txt", outFile )
        self.assertTrue( os.path.exists( outFile ) )
        exp = "line11\nline12\nline13line21\nline22\nline23\nline31\nline32\nline33"
        obs = FileUtils.getFileContent( [ outFile ] )
        self.assertEqual( exp, obs )
        for f in lFiles:
            os.remove( f )
        os.remove(outFile)
            
    def test_catFilesByPattern_with_headers( self ):
        f = open("dummy1.txt", "w")
        f.write("line11\n")
        f.write("line12\n")
        f.write("line13\n")
        f.close()
        f = open("dummy2.txt", "w")
        f.write("line21\n")
        f.write("line22\n")
        f.write("line23\n")
        f.close()
        f = open("dummy3.txt", "w")
        f.write("line31\n")
        f.write("line32\n")
        f.write("line33\n")
        f.close()
        lFiles = [ "dummy1.txt", "dummy2.txt", "dummy3.txt" ]
        outFile = "concatFiles.txt"
        FileUtils.catFilesByPattern( "dummy*.txt", outFile, skipHeaders = True)
        self.assertTrue( os.path.exists( outFile ) )
        exp = "line12\nline13\nline22\nline23\nline32\nline33\n"
        obs = FileUtils.getFileContent( [ outFile ] )
        self.assertEqual( exp, obs )
        for f in lFiles:
            os.remove( f )
        os.remove(outFile)
            
    def test_catFilesByPattern_with_separator( self ):
        f = open("dummy1.txt", "w")
        f.write("line11\n")
        f.write("line12\n")
        f.write("line13")
        f.close()
        f = open("dummy2.txt", "w")
        f.write("line21\n")
        f.write("line22\n")
        f.write("line23\n")
        f.close()
        f = open("dummy3.txt", "w")
        f.write("line31\n")
        f.write("line32\n")
        f.write("line33")
        f.close()
        lFiles = [ "dummy1.txt", "dummy2.txt", "dummy3.txt" ]
        outFile = "concatFiles.txt"
        FileUtils.catFilesByPattern( "dummy*.txt", outFile, separator = "\n+------------+\n")
        self.assertTrue( os.path.exists( outFile ) )
        exp = "line11\nline12\nline13\n+------------+\nline21\nline22\nline23\n\n+------------+\nline31\nline32\nline33"
        obs = FileUtils.getFileContent( [ outFile ] )
        self.assertEqual( exp, obs )
        for f in lFiles:
            os.remove( f )
        os.remove(outFile)
            
    def test_catFilesByPattern_with_headers_and_separator( self ):
        f = open("dummy1.txt", "w")
        f.write("line11\n")
        f.write("line12\n")
        f.write("line13\n")
        f.close()
        f = open("dummy2.txt", "w")
        f.write("line21\n")
        f.write("line22\n")
        f.write("line23\n")
        f.close()
        f = open("dummy3.txt", "w")
        f.write("line31\n")
        f.write("line32\n")
        f.write("line33\n")
        f.close()
        lFiles = [ "dummy1.txt", "dummy2.txt", "dummy3.txt" ]
        outFile = "concatFiles.txt"
        FileUtils.catFilesByPattern( "dummy*.txt", outFile, separator = "\n+------------+\n", skipHeaders = True)
        self.assertTrue( os.path.exists( outFile ) )
        exp = "line12\nline13\n\n+------------+\nline22\nline23\n\n+------------+\nline32\nline33\n"
        obs = FileUtils.getFileContent( [ outFile ] )
        self.assertEqual( exp, obs )
        for f in lFiles:
            os.remove( f )
        os.remove(outFile)
            
    def test_isRessourceExists_exists(self): 
        f = open("dummyFile.txt", "w")
        f.close()
        self.assertTrue(FileUtils.isRessourceExists("dummyFile.txt"))
        os.system("rm dummyFile.txt")
            
    def test_isRessourceExists_not_exists(self):
        self.assertFalse(FileUtils.isRessourceExists("dummyFile.txt"))
            
    def test_isEmpty_empty( self ):
        file = "dummyFile_%s" % ( self._uniqId )
        os.system( "touch %s" % ( file ) )
        self.assertTrue( FileUtils.isEmpty( file ) )
        os.remove( file )
        
    def test_isEmpty_non_empty( self ):
        file = "dummyFile_%s" % ( self._uniqId )
        fileHandler = open( file, "w" )
        fileHandler.write( "line1\n" )
        fileHandler.close()
        self.assertFalse( FileUtils.isEmpty( file ) )
        os.remove( file )
        
    def test_are2FilesIdentical_true( self ):
        f = open("dummy1.txt", "w")
        f.write("line11\n")
        f.close()
        f = open("dummy2.txt", "w")
        f.write("line11\n")
        f.close()
        self.assertTrue( FileUtils.are2FilesIdentical( "dummy1.txt", "dummy2.txt" ) )
        for f in [ "dummy1.txt", "dummy2.txt" ]:
            os.remove( f )
            
    def test_are2FilesIdentical_false( self ):
        f = open("dummy1.txt", "w")
        f.write("line11\n")
        f.close()
        f = open("dummy2.txt", "w")
        f.write("line21\n")
        f.close()
        self.assertFalse( FileUtils.are2FilesIdentical( "dummy1.txt", "dummy2.txt" ) )
        for f in [ "dummy1.txt", "dummy2.txt" ]:
            os.remove( f )
            
    def test_getFileContent( self ):
        inFile = "dummyInFile_%s" % ( self._uniqId )
        inFileHandler = open( inFile, "w" )
        inFileHandler.write( "zgdfet\n" )
        inFileHandler.write( "agdfet\n" )
        inFileHandler.close()
        exp = "zgdfet\n" + "agdfet\n"
        obs = FileUtils.getFileContent( [ inFile ] )
        self.assertEquals( exp, obs )
        os.remove( inFile )
        
    def test_sortFileContent( self ):
        inFile = "dummyInFile_%s" % ( self._uniqId )
        inFileHandler = open( inFile, "w" )
        inFileHandler.write( "zgdfet\n" )
        inFileHandler.write( "agdfet\n" )
        inFileHandler.close()
        
        expFile = "dummyExpFile_%s" % ( self._uniqId )
        expFileHandler = open( expFile, "w" )
        expFileHandler.write( "agdfet\n" )
        expFileHandler.write( "zgdfet\n" )
        expFileHandler.close()
        
        FileUtils.sortFileContent( inFile )
        
        self.assertTrue( FileUtils.are2FilesIdentical( expFile, inFile ) )
        for f in [ inFile, expFile ]:
            os.remove( f )
            
    def test_removeFilesByPattern_prefix( self ):
        fileName1 = "filetest.fa"
        fileName2 = "test.fa.Nstretch.map"
        fileName3 = "test.fa_cut"
        os.system("touch %s" % fileName1)
        os.system("touch %s" % fileName2)
        os.system("touch %s" % fileName3)
        FileUtils.removeFilesByPattern("test*")
        self.assertTrue(os.path.exists(fileName1))
        self.assertFalse(os.path.exists(fileName2))
        self.assertFalse(os.path.exists(fileName3))
        os.remove(fileName1)
            
    def test_removeFilesByPattern_suffix( self ):
        fileName1 = "filetest"
        fileName2 = "test.fa.Nstretch.map"
        fileName3 = "test.fa_cut"
        os.system("touch %s" % fileName1)
        os.system("touch %s" % fileName2)
        os.system("touch %s" % fileName3)
        FileUtils.removeFilesByPattern("*test")
        self.assertFalse(os.path.exists(fileName1))
        self.assertTrue(os.path.exists(fileName2))
        self.assertTrue(os.path.exists(fileName3))
        os.remove(fileName2)
        os.remove(fileName3)
            
    def test_removeFilesBySuffixList( self ):
        tmpDir = "dummyDir_%s" % ( self._uniqId )
        if not os.path.exists( tmpDir ):
            os.mkdir( tmpDir )
        commonPrefix = "dummyFile_%s"  %( self._uniqId )
        os.system( "touch %s/%s.fa" % ( tmpDir, commonPrefix ) )
        os.system( "touch %s/%s.fa.Nstretch.map" % ( tmpDir, commonPrefix ) )
        os.system( "touch %s/%s.fa_cut" % ( tmpDir, commonPrefix ) )
        lSuffixes = [ ".Nstretch.map", "_cut" ]
        FileUtils.removeFilesBySuffixList( tmpDir, lSuffixes )
        self.assertTrue( os.path.exists( "%s/%s.fa" % ( tmpDir, commonPrefix ) ) )
        self.assertFalse( os.path.exists( "%s/%s.fa.Nstretch.map" % ( tmpDir, commonPrefix ) ) )
        self.assertFalse( os.path.exists( "%s/%s.fa_cut" % ( tmpDir, commonPrefix ) ) )
        shutil.rmtree( tmpDir )
        
    def test_removeRepeatedBlanks( self ):
        inFileName = "dummyWithRepeatedBlanks.dum"
        obsFileName = "dummyWithoutRepeatedBlanks.dum"
        expFileName = "dummyExpWithoutRepeatedBlanks.dum"
        self._writeFileWithRepeatedBlanks( inFileName )
        self._writeFileWithoutRepeatedBlanks( expFileName )
        FileUtils.removeRepeatedBlanks( inFileName, obsFileName )
        self.assertTrue( FileUtils.are2FilesIdentical( expFileName, obsFileName ) )
        for f in [ inFileName, expFileName, obsFileName ]:
            os.remove( f )
            
    def test_RemoveRepeatedBlanks_without_outfileName (self):
        inFileName = "dummyWithRepeatedBlanks.dum"
        expFileName = "dummyExpWithoutRepeatedBlanks.dum"
        obsFileName = inFileName
        self._writeFileWithRepeatedBlanks( inFileName )
        self._writeFileWithoutRepeatedBlanks( expFileName )
        FileUtils.removeRepeatedBlanks( inFileName )
        self.assertTrue( FileUtils.are2FilesIdentical( expFileName, obsFileName ) )
        for f in [ inFileName, expFileName ]:
            os.remove( f )


    def test_fromWindowsToUnixEof( self ):
        inFile = "dummyInFile"
        inFileHandler = open( inFile, "w" )
        inFileHandler.write( "toto\r\n" )
        inFileHandler.close()
        expFile = "dummyExpFile"
        expFileHandler = open( expFile, "w" )
        expFileHandler.write( "toto\n" )
        expFileHandler.close()
        FileUtils.fromWindowsToUnixEof( inFile )
        self.assertTrue( FileUtils.are2FilesIdentical( expFile, inFile ) )
        for f in [ inFile, expFile ]:
            os.remove( f )
            
            
    def test_removeDuplicatedLines( self ):
        inFile = "dummyInFile"
        inFileHandler = open( inFile, "w" )
        inFileHandler.write( "toto\n" )
        inFileHandler.write( "titi\n" )
        inFileHandler.write( "toto\n" )
        inFileHandler.close()
        expFile = "dummyExpFile"
        expFileHandler = open( expFile, "w" )
        expFileHandler.write( "toto\n" )
        expFileHandler.write( "titi\n" )
        expFileHandler.close()
        FileUtils.removeDuplicatedLines( inFile )
        self.assertTrue( FileUtils.are2FilesIdentical( expFile, inFile ) )
        for f in [ inFile, expFile ]:
            os.remove( f )
            
            
    def test_writeLineListInFile( self ):
        inFile = "dummyInFile"
        lLines = [ "toto\n", "titi\n" ]
        expFile = "dummyExpFile"
        expFileHandler = open( expFile, "w" )
        expFileHandler.write( "toto\n" )
        expFileHandler.write( "titi\n" )
        expFileHandler.close()
        FileUtils.writeLineListInFile( inFile, lLines )
        self.assertTrue( FileUtils.are2FilesIdentical( expFile, inFile ) )
        for f in [ inFile, expFile ]:
            os.remove( f )
            
        
    def test_getAbsoluteDirectoryPathList(self):
        currentDir = os.getcwd()
        rootDir = currentDir + "/" + "dummy"
        if os.path.exists( rootDir ):
            shutil.rmtree(rootDir)
        os.mkdir(rootDir)
        
        os.mkdir(rootDir + "/" + "dummyDir1")
        os.mkdir(rootDir + "/" + "dummyDir2")
        
        expLDir = [rootDir + "/" + "dummyDir1", rootDir + "/" + "dummyDir2"]
        obsLDir = FileUtils.getAbsoluteDirectoryPathList(rootDir)
        
        expLDir.sort()
        obsLDir.sort()
        self.assertEquals(expLDir, obsLDir)
        
        shutil.rmtree(rootDir + "/" + "dummyDir1")
        shutil.rmtree(rootDir + "/" + "dummyDir2")
        shutil.rmtree(rootDir)
        
        
    def test_getAbsoluteDirectoryPathList_empty_dir(self):
        currentDir = os.getcwd()
        rootDir = currentDir + "/" + "dummy"
        if os.path.exists( rootDir ):
            shutil.rmtree(rootDir)
        os.mkdir(rootDir)
        
        expLDir = []
        obsLDir = FileUtils.getAbsoluteDirectoryPathList(rootDir)
        
        self.assertEquals(expLDir, obsLDir)
        
        shutil.rmtree(rootDir)
        
        
    def test_getSubListAccordingToPattern_match(self):
        lPath = ["/home/repet/titi", "/pattern/test", "/patter/test", "/_patternic_/test", "/home/patternic/test", "/home/pattern", "pattern", ""]
        
        expL = ["/pattern/test", "/_patternic_/test", "/home/patternic/test", "/home/pattern", "pattern"]
        obsL = FileUtils.getSubListAccordingToPattern(lPath, "pattern")
        
        self.assertEquals(expL, obsL)
        
        
    def test_getSubListAccordingToPattern_not_match(self):
        lPath = ["/home/repet/titi", "/pattern/test", "/patter/test", "/_patternic_/test", "/home/patternic/test", "/home/pattern", "pattern", ""]
        
        expL = ["/home/repet/titi", "/patter/test", ""]
        obsL = FileUtils.getSubListAccordingToPattern(lPath, "pattern", False)

        self.assertEquals(expL, obsL)
        
        
    def test_getFileNamesList(self):
        currentDir = os.getcwd()
        rootDir = currentDir + "/" + "dummy"
        if os.path.exists( rootDir ):
            shutil.rmtree(rootDir)
        os.mkdir(rootDir)
        
        directory = "testDir"
        os.mkdir(rootDir + "/" + directory)
        fileName1 = "dummyFile1.gff"
        fullFileName1 = rootDir + "/" + directory + "/" + fileName1
        file1 = open(fullFileName1, "w")
        file1.close()
        fileName2 = "dummyFile2.gff"
        fullFileName2 = rootDir + "/" + directory + "/" + fileName2
        file2 = open(fullFileName2, "w")
        file2.close()
        
        expLFiles = [fileName1, fileName2]
        obsLFiles = FileUtils.getFileNamesList(rootDir + "/" + directory)
        
        self.assertEquals(expLFiles, sorted(obsLFiles))
        
        shutil.rmtree(rootDir)
        
    def test_getFileNamesList_withPattern(self):
        currentDir = os.getcwd()
        rootDir = currentDir + "/" + "dummy"
        if os.path.exists( rootDir ):
            shutil.rmtree(rootDir)
        os.mkdir(rootDir)
        
        directory = "testDir"
        os.mkdir(rootDir + "/" + directory)
        fileName1 = "dummyFile1.gff"
        fullFileName1 = rootDir + "/" + directory + "/" + fileName1
        file1 = open(fullFileName1, "w")
        file1.close()
        fileName2 = "dummyFile2.gff"
        fullFileName2 = rootDir + "/" + directory + "/" + fileName2
        file2 = open(fullFileName2, "w")
        file2.close()
        
        expLFiles = [fileName1]
        obsLFiles = FileUtils.getFileNamesList(rootDir + "/" + directory, "dummyFile1.*")
        
        self.assertEquals(expLFiles, obsLFiles)
        
        shutil.rmtree(rootDir)
        
    def test_getFileNamesList_empty_dir(self):
        currentDir = os.getcwd()
        rootDir = currentDir + "/" + "dummy"
        os.mkdir(rootDir)
    
        directory = "testDir"
        os.mkdir(rootDir + "/" + directory)
        
        expLFiles = []
        obsLFiles = FileUtils.getFileNamesList(rootDir + "/" + directory)
        
        self.assertEquals(expLFiles, obsLFiles)
        
        shutil.rmtree(rootDir)
        
    def test_getMd5SecureHash( self ):
        if "hashlib" in sys.modules:
            inFile = "dummyInFile"
            inFileHandler = open( inFile, "w" )
            inFileHandler.write( "DLZIH17T63B;?" )
            inFileHandler.close()
            exp = "50d1e2ded8f03881f940f70226e2b986"
            obs = FileUtils.getMd5SecureHash( inFile )
            self.assertEqual( exp, obs )
            os.remove( inFile )
            
    def test_catFilesOfDir(self):        
        currentDir = os.getcwd()
        rootDir = currentDir + "/" + "dummy"
        if os.path.exists( rootDir ):
            shutil.rmtree(rootDir)
        os.mkdir(rootDir)
        
        directory = "testDir"
        os.mkdir(rootDir + "/" + directory)
        fileName1 = "dummyFile1.gff"
        fullFileName1 = rootDir + "/" + directory + "/" + fileName1
        file1 = open(fullFileName1, "w")
        file1.write("file1\n")
        file1.close()
        fileName2 = "dummyFile2.gff"
        fullFileName2 = rootDir + "/" + directory + "/" + fileName2
        file2 = open(fullFileName2, "w")
        file2.write("file2\n")
        file2.close()
        obsFile = "obsFile"
        expFile = "expFile"
        expF = open(expFile, "w")
        expF.write("file1\nfile2\n")
        expF.close()
        FileUtils.catFilesOfDir(rootDir + "/" + directory, obsFile)
        self.assertTrue(FileUtils.are2FilesIdentical(expFile, obsFile))
        
        shutil.rmtree(rootDir)
        os.remove(expFile)
        os.remove(obsFile)
        
    def test_isSizeNotNull_True(self):
        file = "dummyExpFile"
        fileHandler = open( file, "w" )
        fileHandler.write( "toto\n" )
        fileHandler.write( "titi\n" )
        fileHandler.close()
        obsSize = FileUtils.isSizeNotNull(file)
        self.assertTrue(obsSize)
        os.remove(file)
        
    def test_isSizeNotNull_False(self):
        file = "dummyExpFile"
        fileHandler = open( file, "w" )
        fileHandler.close()
        obsSize = FileUtils.isSizeNotNull(file)
        self.assertFalse(obsSize)
        os.remove(file)
            
    def test_splitFileIntoNFiles_3_files(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"
        obsFile2 = "dummy-2.txt"
        obsFile3 = "dummy-3.txt"
        
        f = open(inputFile, "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3\n")
        f.close()

        exp1 = "line1\n"
        exp2 = "line2\n"
        exp3 = "line3\n"
        
        FileUtils.splitFileIntoNFiles(inputFile, 3)
        
        obs1 = open(obsFile1).read()
        obs2 = open(obsFile2).read()
        obs3 = open(obsFile3).read()
        
        self.assertEqual(exp1, obs1)
        self.assertEqual(exp2, obs2)
        self.assertEqual(exp3, obs3)
        self.assertFalse(FileUtils.isRessourceExists("dummy-4.txt"))
        FileUtils.removeFilesByPattern("dummy*")
            
    def test_splitFileIntoNFiles_2_files(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"
        obsFile2 = "dummy-2.txt"
        
        f = open(inputFile, "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3\n")
        f.close()

        exp1 = "line1\nline2\n"
        exp2 = "line3\n"
        
        FileUtils.splitFileIntoNFiles(inputFile, 2)
        
        obs1 = open(obsFile1).read()
        obs2 = open(obsFile2).read()
        
        self.assertEqual(exp1, obs1)
        self.assertEqual(exp2, obs2)
        self.assertFalse(FileUtils.isRessourceExists("dummy-3.txt"))
        FileUtils.removeFilesByPattern("dummy*")
            
    def test_splitFileIntoNFiles_one_file(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"
        
        f = open(inputFile, "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3\n")
        f.close()

        exp1 = "line1\nline2\nline3\n"
        
        FileUtils.splitFileIntoNFiles(inputFile, 1)
        
        obs1 = open(obsFile1).read()
        
        self.assertEqual(exp1, obs1)
        self.assertFalse(FileUtils.isRessourceExists("dummy-2.txt"))
        FileUtils.removeFilesByPattern("dummy*")
            
    def test_splitFileIntoNFiles_more_file_than_lines(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"
        obsFile2 = "dummy-2.txt"
        obsFile3 = "dummy-3.txt"
        
        f = open(inputFile, "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3\n")
        f.close()

        exp1 = "line1\n"
        exp2 = "line2\n"
        exp3 = "line3\n"
        
        FileUtils.splitFileIntoNFiles(inputFile, 10)
        
        obs1 = open(obsFile1).read()
        obs2 = open(obsFile2).read()
        obs3 = open(obsFile3).read()
        
        self.assertEqual(exp1, obs1)
        self.assertEqual(exp2, obs2)
        self.assertEqual(exp3, obs3)
        self.assertFalse(FileUtils.isRessourceExists("dummy-4.txt"))
        FileUtils.removeFilesByPattern("dummy*")
        
    def test_splitFileIntoNFiles_empty_file(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"

        os.system( "touch %s" % ( inputFile ) )

        exp1 = ""
        
        FileUtils.splitFileIntoNFiles(inputFile, 10)
        
        obs1 = open(obsFile1).read()
        
        self.assertEqual(exp1, obs1)
        self.assertFalse(FileUtils.isRessourceExists("dummy-2.txt"))
        FileUtils.removeFilesByPattern("dummy*")
        
    def test_splitFileIntoNFiles_0_file(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"
        
        f = open(inputFile, "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3\n")
        f.close()
        
        exp1 = "line1\nline2\nline3\n"
        
        FileUtils.splitFileIntoNFiles(inputFile, 0)
        
        obs1 = open(obsFile1).read()
        
        self.assertEqual(exp1, obs1)
        self.assertFalse(FileUtils.isRessourceExists("dummy-2.txt"))
        FileUtils.removeFilesByPattern("dummy*")
        
    def test_splitFileAccordingToLineNumber_3_files(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"
        obsFile2 = "dummy-2.txt"
        obsFile3 = "dummy-3.txt"
        
        f = open(inputFile, "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3\n")
        f.close()

        exp1 = "line1\n"
        exp2 = "line2\n"
        exp3 = "line3\n"
        
        FileUtils.splitFileAccordingToLineNumber(inputFile, 1)
        
        obs1 = open(obsFile1).read()
        obs2 = open(obsFile2).read()
        obs3 = open(obsFile3).read()
        
        self.assertEqual(exp1, obs1)
        self.assertEqual(exp2, obs2)
        self.assertEqual(exp3, obs3)
        self.assertFalse(FileUtils.isRessourceExists("dummy-4.txt"))
        FileUtils.removeFilesByPattern("dummy*")
            
    def test_splitFileAccordingToLineNumber_2_files(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"
        obsFile2 = "dummy-2.txt"
        
        f = open(inputFile, "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3\n")
        f.close()

        exp1 = "line1\nline2\n"
        exp2 = "line3\n"
        
        FileUtils.splitFileAccordingToLineNumber(inputFile, 2)
        
        obs1 = open(obsFile1).read()
        obs2 = open(obsFile2).read()
        
        self.assertEqual(exp1, obs1)
        self.assertEqual(exp2, obs2)
        self.assertFalse(FileUtils.isRessourceExists("dummy-3.txt"))
        FileUtils.removeFilesByPattern("dummy*")
            
    def test_splitFileAccordingToLineNumber_one_file(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"
        
        f = open(inputFile, "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3\n")
        f.close()

        exp1 = "line1\nline2\nline3\n"
        
        FileUtils.splitFileAccordingToLineNumber(inputFile, 3)
        
        obs1 = open(obsFile1).read()
        
        self.assertEqual(exp1, obs1)
        self.assertFalse(FileUtils.isRessourceExists("dummy-2.txt"))
        FileUtils.removeFilesByPattern("dummy*")
            
    def test_splitFileAccordingToLineNumber_more_maxLines_than_lines(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"
        
        f = open(inputFile, "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3\n")
        f.close()

        exp1 = "line1\nline2\nline3\n"
        
        FileUtils.splitFileAccordingToLineNumber(inputFile, 10)
        
        obs1 = open(obsFile1).read()

        self.assertEqual(exp1, obs1)
        self.assertFalse(FileUtils.isRessourceExists("dummy-2.txt"))
        FileUtils.removeFilesByPattern("dummy*")
            
    def test_splitFileAccordingToLineNumber_empty_file(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"

        os.system( "touch %s" % ( inputFile ) )

        exp1 = ""
        
        FileUtils.splitFileAccordingToLineNumber(inputFile, 10)
        
        obs1 = open(obsFile1).read()
        
        self.assertEqual(exp1, obs1)
        self.assertFalse(FileUtils.isRessourceExists("dummy-2.txt"))
        FileUtils.removeFilesByPattern("dummy*")
            
    def test_splitFileAccordingToLineNumber_0_lines(self):
        inputFile = "dummy.txt"
        obsFile1 = "dummy-1.txt"
        
        f = open(inputFile, "w")
        f.write("line1\n")
        f.write("line2\n")
        f.write("line3\n")
        f.close()

        exp1 = "line1\nline2\nline3\n"
        
        FileUtils.splitFileAccordingToLineNumber(inputFile, 0)
        
        obs1 = open(obsFile1).read()
        
        self.assertEqual(exp1, obs1)
        self.assertFalse(FileUtils.isRessourceExists("dummy-2.txt"))
        FileUtils.removeFilesByPattern("dummy*")
    
    def _writeFile( self, fileName ):
        inFile = open(fileName, 'w')
        inFile.write(">Sequence_de_reference\n")
        inFile.write("ATTTTGCAGTCTTATTCGAG-----GCCATTGCT\n")
        inFile.write(">Lignee1_mismatch\n")
        inFile.write("ATTTTGCAGACTTATTCGAG-----GCCATTGCT\n")
        inFile.write(">Lignee2_insertion\n")
        inFile.write("ATTTTGCAGTCTTATTCGAGATTACGCCATTGCT\n")
        inFile.write(">Lignee3_deletion\n")
        inFile.write("A---TGCAGTCTTATTCGAG-----GCCATTGCT\n")
        inFile.close()      
        
    def _writeFileWithEmptyLine( self, fileName ):
        fileWithEmptyLine = open(fileName, 'w')
        fileWithEmptyLine.write(">Sequence_de_reference\n")
        fileWithEmptyLine.write("ATTTTGCAGTCTTATTCGAG-----GCCATTGCT\n")
        fileWithEmptyLine.write("\n\n")
        fileWithEmptyLine.write(">Lignee1_mismatch\n")
        fileWithEmptyLine.write("ATTTTGCAGACTTATTCGAG-----GCCATTGCT\n")
        fileWithEmptyLine.write("\n\n")
        fileWithEmptyLine.write(">Lignee2_insertion\n")
        fileWithEmptyLine.write("ATTTTGCAGTCTTATTCGAGATTACGCCATTGCT\n")
        fileWithEmptyLine.write("\n")
        fileWithEmptyLine.write(">Lignee3_deletion\n")
        fileWithEmptyLine.write("A---TGCAGTCTTATTCGAG-----GCCATTGCT\n")
        fileWithEmptyLine.close() 
        
    def _writeFileWithRepeatedBlanks( self, fileName ):
        fileWithRepeatedBlanks = open(fileName, 'w')
        fileWithRepeatedBlanks.write(">Sequ  ence_de     _reference\n")
        fileWithRepeatedBlanks.write("ATTTT  GCAGTCTT TTCGAG-  ----GCCATT  GCT\n")
        fileWithRepeatedBlanks.close() 
        
    def _writeFileWithoutRepeatedBlanks( self, fileName ):
        fileWithoutRepeatedBlanks = open(fileName, 'w')
        fileWithoutRepeatedBlanks.write(">Sequ ence_de _reference\n")
        fileWithoutRepeatedBlanks.write("ATTTT GCAGTCTT TTCGAG- ----GCCATT GCT\n")
        fileWithoutRepeatedBlanks.close()
        
test_suite = unittest.TestSuite()
test_suite.addTest( unittest.makeSuite( Test_FileUtils ) )
if __name__ == "__main__":
    unittest.TextTestRunner(verbosity=2).run( test_suite )
