#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import re
import sys
from SMART.Java.Python.structure.Interval import Interval
from SMART.Java.Python.structure.Transcript import Transcript
from commons.core.parsing.TranscriptListParser import TranscriptListParser


class GtfParser(TranscriptListParser):
    """A class that parses a GTF file and create a transcript list"""


    def __init__(self, fileName, verbosity = 0):
        super(GtfParser, self).__init__(fileName, verbosity)


    def __del__(self):
        super(GtfParser, self).__del__()


    def getFileFormats():
        return ["gtf", "gtf2"]
    getFileFormats = staticmethod(getFileFormats)


    def skipFirstLines(self):
        pass


    def parseLine(self, line):
        if line[0] == "#":
            return None
        m = re.search(r"^\s*(\S+)\s+(\S+)\s+(\S+)\s+(\d+)\s+(\d+)\s+(\S+)\s+([+-.])\s+(\S+)\s+(\S.*)$", line)
        if m == None:
            raise Exception("\nLine %d '%s' does not have a GTF format\n" % (self.currentLineNb, line))
        interval = Interval()
        interval.setChromosome(m.group(1))
        interval.setName("unnamed transcript")
        interval.setStart(min(int(m.group(4)), int(m.group(5))))
        interval.setEnd(max(int(m.group(4)), int(m.group(5))))
        if m.group(7) == ".":
            interval.setDirection("+")
        else:
            interval.setDirection(m.group(7))
        if m.group(6).isdigit():
            interval.setTagValue("score", m.group(6))
        type = m.group(3)

        if type not in ("transcript", "exon"):
            return None

        remainings = m.group(9).split(";")
        for remaining in remainings:
            remaining = remaining.strip()
            if remaining == "":
                continue
            parts = remaining.split(" ", 1)
            field = parts[0].strip()
            value = " ".join(parts[1:]).strip(" \"")
            if field == "transcript_id":
                interval.setTagValue("ID", value)
            elif field == "gene_name":
                interval.setName(value)
            elif field == "transcript_name":
                interval.setName(value)
            elif field == "exon_number":
                continue
            else:
                try:
                    intValue = int(value)
                    interval.setTagValue(field, intValue)
                except ValueError:
                    interval.setTagValue(field, value)

        self.currentTranscriptAddress = self.previousTranscriptAddress
        if self.currentTranscript == None or interval.getTagValue("ID") != self.currentTranscript.getTagValue("ID"):
            transcript = self.currentTranscript
            self.currentTranscript = Transcript()
            self.currentTranscript.copy(interval)
            self.currentTranscript.setTagValue("feature", "transcript")
            self.previousTranscriptAddress = self.currentAddress
            return transcript
        if type == "exon":
            self.currentTranscript.addExon(interval)
        return None
