# Copyright INRA (Institut National de la Recherche Agronomique)
# http://www.inra.fr
# http://urgi.versailles.inra.fr
#
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software.  You can  use, 
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info". 
#
# As a counterpart to the access to the source code and  rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty  and the software's author,  the holder of the
# economic rights,  and the successive licensors  have only  limited
# liability. 
#
# In this respect, the user's attention is drawn to the risks associated
# with loading,  using,  modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean  that it is complicated to manipulate,  and  that  also
# therefore means  that it is reserved for developers  and  experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or 
# data to be ensured and,  more generally, to use and operate it in the 
# same conditions as regards security. 
#
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.


import unittest
import os
import time
from commons.core.sql.TablePathAdaptator import TablePathAdaptator
from commons.core.coord.Path import Path
from commons.core.coord.Set import Set
from commons.core.sql.DbMySql import DbMySql
from commons.core.coord.Range import Range
from commons.core.coord.PathUtils import PathUtils
from copy import deepcopy

class Test_TablePathAdaptator( unittest.TestCase ):
    
    def setUp( self ):
        self._uniqId = "%s_%s" % ( time.strftime("%Y%m%d%H%M%S") , os.getpid() )
        self._configFileName = "dummyConfigFile_%s" % ( self._uniqId )
        configF = open(self._configFileName, "w" )
        configF.write( "[repet_env]\n" )
        configF.write( "repet_host: %s\n" % ( os.environ["REPET_HOST"] ) )
        configF.write( "repet_user: %s\n" % ( os.environ["REPET_USER"] ) )
        configF.write( "repet_pw: %s\n" % ( os.environ["REPET_PW"] ) )
        configF.write( "repet_db: %s\n" % ( os.environ["REPET_DB"] ) )
        configF.write( "repet_port: %s\n" % ( os.environ["REPET_PORT"] ) )
        configF.close()
        self._db = DbMySql( cfgFileName = self._configFileName )
        self._table = "dummyPathTable_%s" % ( self._uniqId )
        self._tpA = TablePathAdaptator( self._db, self._table )
        
        
    def tearDown( self ):
        self._uniqId = None
        self._db.dropTable( self._table )
        self._db.close()
        self._table = None
        self._tMatchA = None
        os.remove( self._configFileName )
        self._configFileName = ""  
        
        
##################################################################################
################## Tests for methods in ITableMapAdaptator #######################
##################################################################################       
     
    def test_getPathListFromId( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write( "1\tchr1\t1\t6\tTE2\t11\t16\t1e-20\t30\t90.2\n" )
        pathF.write( "2\tchr1\t1001\t1006\tTE2\t11\t16\t1e-20\t30\t90.2\n" )
        pathF.write( "2\tchr1\t1201\t1226\tTE2\t10\t26\t1e-40\t70\t87.2\n" )
        pathF.close()
        p1 = Path()
        p1.setFromString( "2\tchr1\t1001\t1006\tTE2\t11\t16\t1e-20\t30\t90.2\n" )
        p2 = Path()
        p2.setFromString( "2\tchr1\t1201\t1226\tTE2\t10\t26\t1e-40\t70\t87.2\n" )
        lExp = [ p1, p2 ]
        self._db.createTable( self._table, "path", pathFileName )
        lObs = self._tpA.getPathListFromId( 2 )
        self.assertEqual( lObs, lExp )
        os.remove( pathFileName )
        
        
    def test_getPathListFromIdList_empty_id_list( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("3\tchr1\t8\t13\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()
        self._db.createTable( self._table, "path", pathFileName )
        
        expList = []
        obsList = self._tpA.getPathListFromIdList([])
        self.assertEquals(expList, obsList)
        
        os.remove( pathFileName )
        
        
    def test_getPathListFromIdList( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("3\tchr1\t8\t13\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()

        tuple = ("1", "chr1","1", "10","TE2","11","17","1e-20","30","90.2")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("2", "chr1","2", "9","TE2","10","13","1e-20","30","90.2")
        p2 = Path()
        p2.setFromTuple(tuple)

        tuple = ("3", "chr1","8", "13","TE2","11","17","1e-20","30","90.2")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        lObs = self._tpA.getPathListFromIdList((1, 2, 3))
        self.assertEquals(3, len(lObs))
        
        lExp = [p1, p2, p3]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )
        
        
    def test_getPathListFromQuery( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("3\tchr2\t8\t13\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()
        self._db.createTable( self._table, "path", pathFileName )
        path1 = Path()
        path1.setFromString("1\tchr1\t1\t10\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        path2 = Path()
        path2.setFromString("2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        expList = [ path1, path2 ]
        obsList = self._tpA.getPathListFromQuery("chr1")
        self.assertEquals( expList, obsList )
        
        os.remove( pathFileName )
        
        
    def test_getPathListFromQuery_unexisted_query( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("3\tchr2\t8\t13\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()
        self._db.createTable( self._table, "path", pathFileName )
        expList = []
        obsList = self._tpA.getPathListFromQuery("dummy")
        self.assertEquals( expList, obsList )
        
        os.remove( pathFileName )
        
        
    def test_getPathListFromSubject( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE3\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("3\tchr2\t8\t13\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()
        self._db.createTable( self._table, "path", pathFileName )
        path1 = Path()
        path1.setFromString("1\tchr1\t1\t10\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        path3 = Path()
        path3.setFromString("3\tchr2\t8\t13\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        expList = [ path1, path3 ]
        obsList = self._tpA.getPathListFromSubject("TE2")
        self.assertEquals( expList, obsList )

        os.remove( pathFileName )
        
        
    def test_getPathListFromSubject_unexisted_subject( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE3\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("3\tchr2\t8\t13\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()
        self._db.createTable( self._table, "path", pathFileName )
        expList = []
        obsList = self._tpA.getPathListFromSubject("dummy")
        self.assertEquals( expList, obsList )

        os.remove( pathFileName )
        
        
    def test_insert_QryDirSbjDir( self ):
        path2Insert = Path()
        
        rangeQuery2Insert = Range()
        rangeQuery2Insert.seqname = "chr1"
        rangeQuery2Insert.start = 1
        rangeQuery2Insert.end = 10
        
        rangeSubject2Insert = Range()
        rangeSubject2Insert.seqname = "TE2"
        rangeSubject2Insert.start = 11
        rangeSubject2Insert.end = 17
         
        path2Insert.range_query = rangeQuery2Insert
        path2Insert.range_subject = rangeSubject2Insert
        
        path2Insert.identity = 90.2
        path2Insert.score = 30
        path2Insert.e_value = 1e-20
        path2Insert.id = 1
        
        self._db.createTable( self._table, "path" )
        self._tpA.insert( path2Insert )
        
        expPathTuple = ((1L, "chr1", 1L, 10L, "TE2", 11L, 17L, 1e-20, 30L, 90.2),)
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        
        self.assertEquals(expPathTuple, obsPathTuple)
        
        
    def test_insert_BigEvalue( self ):
        path2Insert = Path()
        
        rangeQuery2Insert =  Range()
        rangeQuery2Insert.seqname = "chr1"
        rangeQuery2Insert.start = 1
        rangeQuery2Insert.end = 10
        
        rangeSubject2Insert =  Range()
        rangeSubject2Insert.seqname = "TE2"
        rangeSubject2Insert.start = 11
        rangeSubject2Insert.end = 17
         
        path2Insert.range_query = rangeQuery2Insert
        path2Insert.range_subject = rangeSubject2Insert
        
        path2Insert.identity = 90.2
        path2Insert.score = 30.0
        path2Insert.e_value = 1e-300
        path2Insert.id = 1
        
        self._db.createTable( self._table, "path" )
        self._tpA.insert( path2Insert )
        
        expPathTuple = ((1L, "chr1", 1L, 10L, "TE2", 11L, 17L, 1e-300, 30L, 90.2),)
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        
        self.assertEquals(expPathTuple, obsPathTuple)
        
        
    def test_insert_QryRevSbjDir( self ):
        path2Insert = Path()
        
        rangeQuery2Insert =  Range()
        rangeQuery2Insert.seqname = "chr1"
        rangeQuery2Insert.start = 10
        rangeQuery2Insert.end = 1
        
        rangeSubject2Insert =  Range()
        rangeSubject2Insert.seqname = "TE2"
        rangeSubject2Insert.start = 11
        rangeSubject2Insert.end = 17
         
        path2Insert.range_query = rangeQuery2Insert
        path2Insert.range_subject = rangeSubject2Insert
        
        path2Insert.identity = 90.2
        path2Insert.score = 30
        path2Insert.e_value = 1e-216
        path2Insert.id = 1
        
        self._db.createTable( self._table, "path" )
        self._tpA.insert( path2Insert )
        
        expPathTuple = ((1L, "chr1", 1L, 10L, "TE2", 17L, 11L, 1e-216, 30L, 90.2),)
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        
        self.assertEquals(expPathTuple, obsPathTuple)
        
        
    def test_insertList(self):
        p1 = Path()
        p1.setFromString( "1\tchr1\t1\t10\tTE1\t11\t17\t1e-20\t30\t90.2\n" )
        p2 = Path()
        p2.setFromString( "2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n" )
        p3 = Path()
        p3.setFromString( "2\tchr1\t12\t19\tTE2\t15\t22\t1e-10\t40\t94.2\n" )
        p4 = Path()
        p4.setFromString( "3\tchr2\t8\t13\tTE1\t11\t17\t1e-20\t30\t90.2\n" )
        lPath = [ p1, p2, p3, p4]
        self._db.createTable( self._table, "path" )
        self._tpA.insertList(lPath)
        
        tuple1 = (1L, "chr1", 1L, 10L, "TE1", 11L, 17L, 1e-20, 30L, 90.2)
        tuple2 = (2L, "chr1", 2L, 9L, "TE2", 10L, 13L, 1e-20, 30L, 90.2)
        tuple3 = (2L, "chr1", 12L, 19L, "TE2", 15L, 22L, 1e-10, 40L, 94.2)
        tuple4 = (3L, "chr2", 8L, 13L, "TE1", 11L, 17L, 1e-20, 30L, 90.2)
        expPathTuple = ( tuple1, tuple2, tuple3, tuple4)
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        
        self.assertEquals(expPathTuple, obsPathTuple)
        
        
    def test_getIdListFromQuery( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromString( "1\tchr1\t1\t10\tTE1\t11\t17\t1e-20\t30\t90.2\n" )
        p2a = Path()
        p2a.setFromString( "2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n" )
        p2b = Path()
        p2b.setFromString( "2\tchr1\t12\t19\tTE2\t15\t22\t1e-10\t40\t94.2\n" )
        p3 = Path()
        p3.setFromString( "3\tchr2\t8\t13\tTE1\t11\t17\t1e-20\t30\t90.2\n" )
        for p in [ p1, p2a, p2b, p3 ]: self._tpA.insert( p )
        lExp = [ 1, 2 ]
        lObs = self._tpA.getIdListFromQuery( "chr1" )
        self.assertEqual( lObs, lExp )
        
        
    def test_getIdListFromSubject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromString( "1\tchr1\t1\t10\tTE1\t11\t17\t1e-20\t30\t90.2\n" )
        p2a = Path()
        p2a.setFromString( "2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n" )
        p2b = Path()
        p2b.setFromString( "2\tchr1\t12\t19\tTE2\t15\t22\t1e-10\t40\t94.2\n" )
        p3 = Path()
        p3.setFromString( "3\tchr2\t8\t13\tTE1\t11\t17\t1e-20\t30\t90.2\n" )
        for p in [ p1, p2a, p2b, p3 ]: self._tpA.insert( p )
        lExp = [ 2 ]
        lObs = self._tpA.getIdListFromSubject( "TE2" )
        self.assertEqual( lObs, lExp )
        
        
    def test_getIdList( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromString( "1\tchr1\t1\t10\tTE1\t11\t17\t1e-20\t30\t90.2\n" )
        p2 = Path()
        p2.setFromString( "2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n" )
        p3 = Path()
        p3.setFromString( "2\tchr1\t12\t19\tTE2\t15\t22\t1e-10\t40\t94.2\n" )
        p4 = Path()
        p4.setFromString( "3\tchr2\t8\t13\tTE1\t11\t17\t1e-20\t30\t90.2\n" )
        lPath = [ p1, p2, p3, p4]
        self._tpA.insertList(lPath)
        expList = [ 1, 2, 3 ]
        obsList = self._tpA.getIdList()
        self.assertEqual( expList, obsList )
        
        
    def test_getSubjectList( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE3\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t12\t19\tTE2\t15\t22\t1e-10\t40\t94.2\n")
        pathF.write("3\tchr2\t8\t13\tTE1\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()
        self._db.createTable( self._table, "path", pathFileName )
        expList = [ "TE1", "TE2", "TE3" ]
        obsList = self._tpA.getSubjectList()
        self.assertEqual( expList, obsList )
        os.remove( pathFileName )
        
        
    def test_getSubjectList_empty_table( self ):
        self._db.createTable( self._table, "path" )
        expList = []
        obsList = self._tpA.getSubjectList()
        self.assertEqual( obsList, expList )
        
        
    def test_getQueryList( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE3\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t12\t19\tTE2\t15\t22\t1e-10\t40\t94.2\n")
        pathF.write("3\tchr2\t8\t13\tTE1\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()
        self._db.createTable( self._table, "path", pathFileName )
        expList = [ "chr1", "chr2" ]
        obsList = self._tpA.getQueryList()
        self.assertEqual( expList, obsList )
        os.remove( pathFileName )
        
        
    def test_getSubjectListFromQuery( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE3\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t12\t19\tTE2\t15\t22\t1e-10\t40\t94.2\n")
        pathF.write("3\tchr2\t8\t13\tTE1\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()
        self._db.createTable( self._table, "path", pathFileName )
        expList = [ "TE3", "TE2" ]
        obsList = self._tpA.getSubjectListFromQuery( "chr1" )
        self.assertEqual( expList, obsList )
        os.remove( pathFileName )
        
        
    def test_getSubjectListFromQuery_with_unexisted_query( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE3\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t12\t19\tTE2\t15\t22\t1e-10\t40\t94.2\n")
        pathF.write("3\tchr2\t8\t13\tTE1\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()
        self._db.createTable( self._table, "path", pathFileName )
        expList = []
        obsList = self._tpA.getSubjectListFromQuery( "chr3" )
        self.assertEqual( expList, obsList )
        os.remove( pathFileName )
        
        
    def test_getListOfAllPaths( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromString( "1\tchr1\t1\t10\tTE3\t11\t17\t1e-20\t30\t85.2\n" )
        p2a = Path()
        p2a.setFromString( "2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.5\n" )
        p2b = Path()
        p2b.setFromString( "2\tchr1\t12\t19\tTE2\t15\t22\t1e-10\t40\t89.5\n" )
        lPaths = [ p1, p2a, p2b ]
        self._tpA.insertList( lPaths )
        expList = [ p1, p2a, p2b ]
        obsList = self._tpA.getListOfAllPaths()
        self.assertEqual( expList, obsList )
        
        
    def test_getListOfAllPaths_empty_table( self ):
        self._db.createTable( self._table, "path" )
        expList = []
        obsList = self._tpA.getListOfAllPaths()
        self.assertEqual( expList, obsList )
        
        
    def test_getPathListWithDirectQueryDirectSubjectFromQuerySubject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromString( "1\tchr1\t1\t100\tTE1\t11\t110\t1e-20\t130\t85.2\n" )
        p2 = Path()
        p2.setFromString( "2\tchr2\t1\t100\tTE1\t11\t110\t1e-20\t130\t85.2\n" )  # different query
        p3 = Path()
        p3.setFromString( "3\tchr1\t1\t100\tTE2\t11\t110\t1e-20\t130\t85.2\n" )  # different subject
        p4 = Path()
        p4.setFromString( "4\tchr1\t100\t1\tTE1\t11\t110\t1e-20\t130\t85.2\n" )  # query on reverse strand
        p5 = Path()
        p5.setFromString( "5\tchr1\t1\t100\tTE1\t110\t11\t1e-20\t130\t85.2\n" )  # subject on reverse strand
        p6 = Path()
        p6.setFromString( "6\tchr1\t301\t400\tTE1\t11\t110\t1e-20\t130\t85.2\n" )  # further along the query
        for p in [ p1, p2, p3, p4, p5, p6 ]: self._tpA.insert( p )
        expList = [ p1, p6 ]
        obsList = self._tpA.getPathListWithDirectQueryDirectSubjectFromQuerySubject( "chr1", "TE1" )
        self.assertEqual( expList, obsList )
        
        
    def test_getPathListWithDirectQueryReverseSubjectFromQuerySubject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromString( "1\tchr1\t1\t100\tTE1\t110\t11\t1e-20\t130\t85.2\n" )
        p2 = Path()
        p2.setFromString( "2\tchr2\t1\t100\tTE1\t110\t11\t1e-20\t130\t85.2\n" )  # different query
        p3 = Path()
        p3.setFromString( "3\tchr1\t1\t100\tTE2\t110\t11\t1e-20\t130\t85.2\n" )  # different subject
        p4 = Path()
        p4.setFromString( "4\tchr1\t100\t1\tTE1\t110\t11\t1e-20\t130\t85.2\n" )  # query on reverse strand
        p5 = Path()
        p5.setFromString( "5\tchr1\t1\t100\tTE1\t11\t110\t1e-20\t130\t85.2\n" )  # subject on direct strand
        p6 = Path()
        p6.setFromString( "6\tchr1\t301\t400\tTE1\t110\t11\t1e-20\t130\t85.2\n" )  # further along the query
        for p in [ p1, p2, p3, p4, p5, p6 ]: self._tpA.insert( p )
        lExp = [ p1, p6 ]
        lObs = self._tpA.getPathListWithDirectQueryReverseSubjectFromQuerySubject( "chr1", "TE1" )
        self.assertEqual( lObs, lExp )
        
        
    def test_isEmpty( self ):
        self._db.createTable( self._table, "path" )
        obs = self._tpA.isEmpty()
        self.assertTrue( obs )
        p = Path()
        p.setFromTuple( ( "1", "qry1", "1", "100", "sbj1", "1", "100", "1e-32", "164", "97.5" ) )
        self._tpA.insert( p )
        obs = self._tpA.isEmpty()
        self.assertFalse( obs )
        
        
    def test_getNbPathsFromQuery( self ):
        self._db.createTable( self._table, "path" )
        p1a = Path()
        p1a.setFromTuple( ( "1", "qry1", "1", "100", "sbj1", "1", "100", "1e-32", "164", "97.5" ) )
        p1b = Path()
        p1b.setFromTuple( ( "1", "qry1", "111", "200", "sbj1", "101", "190", "1e-32", "164", "97.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "2001", "2200", "sbj3", "1", "200", "1e-76", "247", "96.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "1", "350", "sbj1", "1", "350", "1e-43", "624", "98.1" ) )
        lPaths = [ p1a, p1b, p2, p3 ]
        self._tpA.insertList( lPaths )
        expNb = 3
        obsNb = self._tpA.getNbPathsFromQuery( "qry1" )
        self.assertEqual( expNb, obsNb )
        
        
    def test_getNbPathsFromQuery_unexisted_query( self ):
        self._db.createTable( self._table, "path" )
        p1a = Path()
        p1a.setFromTuple( ( "1", "qry1", "1", "100", "sbj1", "1", "100", "1e-32", "164", "97.5" ) )
        p1b = Path()
        p1b.setFromTuple( ( "1", "qry1", "111", "200", "sbj1", "101", "190", "1e-32", "164", "97.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "2001", "2200", "sbj3", "1", "200", "1e-76", "247", "96.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "1", "350", "sbj1", "1", "350", "1e-43", "624", "98.1" ) )
        lPaths = [ p1a, p1b, p2, p3 ]
        self._tpA.insertList( lPaths )
        expNb = 0
        obsNb = self._tpA.getNbPathsFromQuery( "qry3" )
        self.assertEqual( expNb, obsNb )
        
        
    def test_getNbPathsFromSubject( self ):
        self._db.createTable( self._table, "path" )
        p1a = Path()
        p1a.setFromTuple( ( "1", "qry1", "1", "100", "sbj1", "1", "100", "1e-32", "164", "97.5" ) )
        p1b = Path()
        p1b.setFromTuple( ( "1", "qry1", "111", "200", "sbj1", "101", "190", "1e-32", "164", "97.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "2001", "2200", "sbj3", "1", "200", "1e-76", "247", "96.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "1", "350", "sbj1", "1", "350", "1e-43", "624", "98.1" ) )
        lPaths = [ p1a, p1b, p2, p3 ]
        self._tpA.insertList( lPaths )
        expNb = 3
        obsNb = self._tpA.getNbPathsFromSubject( "sbj1" )
        self.assertEqual( expNb, obsNb )
        
        
    def test_getNbPathsFromSubject_unexisted_subject( self ):
        self._db.createTable( self._table, "path" )
        p1a = Path()
        p1a.setFromTuple( ( "1", "qry1", "1", "100", "sbj1", "1", "100", "1e-32", "164", "97.5" ) )
        p1b = Path()
        p1b.setFromTuple( ( "1", "qry1", "111", "200", "sbj1", "101", "190", "1e-32", "164", "97.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "2001", "2200", "sbj3", "1", "200", "1e-76", "247", "96.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "1", "350", "sbj1", "1", "350", "1e-43", "624", "98.1" ) )
        lPaths = [ p1a, p1b, p2, p3 ]
        self._tpA.insertList( lPaths )
        expNb = 0
        obsNb = self._tpA.getNbPathsFromSubject( "qry1" )
        self.assertEqual( expNb, obsNb )
        
        
    def test_getNbIds( self ):
        self._db.createTable( self._table, "path" )
        p1a = Path()
        p1a.setFromTuple( ( "1", "qry1", "1", "100", "sbj1", "1", "100", "1e-32", "164", "97.5" ) )
        p1b = Path()
        p1b.setFromTuple( ( "1", "qry1", "111", "200", "sbj1", "101", "190", "1e-32", "164", "97.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "2001", "2200", "sbj3", "1", "200", "1e-76", "247", "96.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "1", "350", "sbj1", "1", "350", "1e-43", "624", "98.1" ) )
        for p in [ p1a, p1b, p2, p3 ]: self._tpA.insert( p )
        exp = 3
        obs = self._tpA.getNbIds()
        self.assertEqual( obs, exp )
        
        
    def test_getNbIdsFromSubject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "1", "150", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )

        expNb = 2
        obsNb = self._tpA.getNbIdsFromSubject( "sbj1" )
        self.assertEqual( expNb, obsNb )
        
        
    def test_getNbIdsFromSubject_unexisted_subject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "1", "150", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )

        expNb = 0
        obsNb = self._tpA.getNbIdsFromSubject( "sbj2" )
        self.assertEqual( expNb, obsNb ) 
        
        
    def test_getNbIdsFromQuery( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )

        expNb = 2
        obsNb = self._tpA.getNbIdsFromQuery( "qry1" )
        self.assertEqual( expNb, obsNb )
        
        
    def test_getNbIdsFromQuery_unexisted_query( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )

        expNb = 0
        obsNb = self._tpA.getNbIdsFromQuery( "qry2" )
        self.assertEqual( expNb, obsNb )
        
        
    def test_getPathListIncludedInQueryCoord_included( self ):
        self._db.createTable( self._table, "path" )
        p = Path()
        p.setFromTuple( ( "1", "qry1", "123", "184", "sbj1", "1", "63", "0.0", "75", "97.5" ) )
        self._tpA.insert( p )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        expList = [ p ]
        obsList = self._tpA.getPathListIncludedInQueryCoord( "qry1", 100, 200 )
        self.assertEqual( expList, obsList )
        
        
    def test_getPathListIncludedInQueryCoord_included_start_higher_than_end( self ):
        self._db.createTable( self._table, "path" )
        p = Path()
        p.setFromTuple( ( "1", "qry1", "123", "184", "sbj1", "1", "63", "0.0", "75", "97.5" ) )
        self._tpA.insert( p )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        expList = [ p ]
        obsList = self._tpA.getPathListIncludedInQueryCoord( "qry1", 200, 100 )
        self.assertEqual( expList, obsList )
        
        
    def test_getPathListIncludedInQueryCoord_overlapping( self ):
        self._db.createTable( self._table, "path" )
        p = Path()
        p.setFromTuple( ( "1", "qry1", "83", "184", "sbj1", "1", "103", "0.0", "137", "96.5" ) )
        self._tpA.insert( p )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        expList = []
        obsList = self._tpA.getPathListIncludedInQueryCoord( "qry1", 100, 200 )
        self.assertEqual( expList, obsList )
        
        
    def test_getPathListOverlappingQueryCoord(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "83", "184", "sbj1", "1", "103", "0.0", "137", "96.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry1", "1", "350", "sbj1", "1", "350", "1e-43", "624", "98.1" ) )
        lPath = [ p1, p2, p3 ]
        self._tpA.insertList(lPath)
        expList = [ p1, p3 ]
        obsList = self._tpA.getPathListOverlappingQueryCoord( "qry1", 100, 200 )
        self.assertEqual( expList, obsList )
        
        
    def test_getPathListOverlappingQueryCoord_no_overlapping_and_start_higher_than_end(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "83", "184", "sbj1", "1", "103", "0.0", "137", "96.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "1", "350", "sbj1", "1", "350", "1e-43", "624", "98.1" ) )
        lPath = [ p1, p2, p3 ]
        self._tpA.insertList(lPath)
        expList = []
        obsList = self._tpA.getPathListOverlappingQueryCoord( "qry1", 80, 1 )
        self.assertEqual( expList, obsList )
        
        
    def test_getPathListOverlappingQueryCoord_withChains(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "83", "184", "sbj1", "1", "103", "0.0", "137", "96.5" ) )
        p2a = Path()
        p2a.setFromTuple( ( "2", "qry1", "1", "150", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        p2b = Path()
        p2b.setFromTuple( ( "2", "qry1", "251", "350", "sbj1", "151", "250", "1e-43", "624", "98.1" ) )
        lPath = [ p1, p2a, p2b ]
        self._tpA.insertList(lPath)
        expList = [ p1, p2a ]
        obsList = self._tpA.getPathListOverlappingQueryCoord( "qry1", 100, 200 )
        self.assertEqual( expList, obsList )
        
        
    def test_getChainListOverlappingQueryCoord(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "83", "184", "sbj1", "1", "103", "0.0", "137", "96.5" ) )
        p2a = Path()
        p2a.setFromTuple( ( "2", "qry1", "1", "150", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        p2b = Path()
        p2b.setFromTuple( ( "2", "qry1", "251", "350", "sbj1", "151", "250", "1e-43", "624", "98.1" ) )
        lPath = [ p1, p2a, p2b ]
        self._tpA.insertList(lPath)
        expList = [ p1, p2a, p2b ]
        obsList = self._tpA.getChainListOverlappingQueryCoord( "qry1", 100, 200 )
        self.assertEqual( expList, obsList )
        
        
    def test_getSetListOverlappingQueryCoord(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "83", "184", "sbj1", "1", "103", "0.0", "137", "96.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry1", "1", "350", "sbj2", "1", "350", "1e-43", "624", "98.1" ) )
        lPath = [ p1, p2, p3 ]
        self._tpA.insertList(lPath)
        
        set1 = Set()
        set1.setFromTuple( ( "1", "sbj1", "qry1", "83", "184" ) )
        set3 = Set()
        set3.setFromTuple( ( "3", "sbj2", "qry1", "1", "350" ) )
        expList = [ set1, set3 ]
        obsList = self._tpA.getSetListOverlappingQueryCoord( "qry1", 100, 200 )
        self.assertEqual( expList, obsList )
        
        
    def test_getSetListOverlappingQueryCoord_no_overlapping_and_start_higher_than_end(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "83", "184", "sbj1", "1", "103", "0.0", "137", "96.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "1", "350", "sbj1", "1", "350", "1e-43", "624", "98.1" ) )
        lPath = [ p1, p2, p3 ]
        self._tpA.insertList(lPath)
        expList = []
        obsList = self._tpA.getSetListOverlappingQueryCoord( "qry1", 80, 1 )
        self.assertEqual( expList, obsList )
        
        
    def test_getSetListIncludedInQueryCoord(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "102", "184", "sbj1", "1", "103", "0.0", "137", "96.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry1", "1", "350", "sbj2", "1", "350", "1e-43", "624", "98.1" ) )
        lPath = [ p1, p2, p3 ]
        self._tpA.insertList(lPath)
        
        set1 = Set()
        set1.setFromTuple( ( "1", "sbj1", "qry1", "102", "184" ) )
        expList = [ set1 ]
        obsList = self._tpA.getSetListIncludedInQueryCoord( "qry1", 100, 200 )
        self.assertEqual( expList, obsList )
        
        
    def test_getSetListIncludedInQueryCoord_no_including_and_start_higher_than_end(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "83", "184", "sbj1", "1", "103", "0.0", "137", "96.5" ) )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0", "187", "97.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "1", "350", "sbj1", "1", "350", "1e-43", "624", "98.1" ) )
        lPath = [ p1, p2, p3 ]
        self._tpA.insertList(lPath)
        expList = []
        obsList = self._tpA.getSetListIncludedInQueryCoord( "qry1", 80, 1 )
        self.assertEqual( expList, obsList )
        
        
    def test_getPathListSortedByQueryCoord( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "3", "qry2", "101", "200", "sbj3", "1", "100", "0.0", "137", "96.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "151", "500", "sbj1", "1", "350", "0.0", "137", "96.5" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "1", "qry1", "1", "200", "sbj3", "1", "200", "0.0", "137", "96.5" ) )
        self._tpA.insert( p3 )
        
        expList = [ p3, p2, p1 ]
        obsList = self._tpA.getPathListSortedByQueryCoord()
        self.assertEqual( expList, obsList )
        
        
    def test_getPathListSortedByQueryCoordFromQuery( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "3", "qry2", "101", "200", "sbj3", "1", "100", "0.0", "137", "96.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "151", "500", "sbj1", "1", "350", "0.0", "137", "96.5" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "1", "qry1", "1", "200", "sbj3", "1", "200", "0.0", "137", "96.5" ) )
        self._tpA.insert( p3 )
        
        expList = [ p3, p2 ]
        obsList = self._tpA.getPathListSortedByQueryCoordFromQuery( "qry1" )
        self.assertEqual( expList, obsList )
        
    def test_getPathListSortedByQueryCoordAndScoreFromQuery( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "3", "qry2", "101", "200", "sbj3", "1", "100", "0.0", "137", "96.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "151", "500", "sbj1", "1", "350", "0.0", "137", "96.5" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "1", "qry1", "1", "200", "sbj3", "1", "200", "0.0", "137", "96.5" ) )
        self._tpA.insert( p3 )
        p4 = Path()
        p4.setFromTuple( ( "4", "qry1", "1", "200", "sbj3", "1", "200", "0.0", "200", "96.5" ) )
        self._tpA.insert( p4 )
        
        expList = [  p3, p4, p2 ]
        obsList = self._tpA.getPathListSortedByQueryCoordAndScoreFromQuery( "qry1" )
        self.assertEqual( expList, obsList )
        
        
    def test_getCumulLengthFromSubject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "1", "150", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )

        exp = 250
        obs = self._tpA.getCumulLengthFromSubject( "sbj1" )
        self.assertEqual( obs, exp )
        
        
    def test_getCumulLengthFromSubject_with_no_subject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "1", "150", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )

        exp = 0
        obs = self._tpA.getCumulLengthFromSubject( "sbj2" )
        self.assertEqual( obs, exp )    
        
        
    def test_getChainLengthListFromSubject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "1", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "2", "qry1", "801", "900", "sbj1", "1", "100", "0.0", "187", "97.2" ) )
        self._tpA.insert( p3 )
        p4 = Path()
        p4.setFromTuple( ( "3", "qry1", "1900", "1801", "sbj1", "1", "100", "0.0", "187", "97.2" ) )
        self._tpA.insert( p4 )
        p5 = Path()
        p5.setFromTuple( ( "4", "qry1", "1801", "1900", "sbj1", "100", "1", "0.0", "187", "97.2" ) )
        self._tpA.insert( p5 )
        
        expList = [ 250, 100, 100, 100 ]
        obsList = self._tpA.getChainLengthListFromSubject( "sbj1" )
        self.assertEqual( expList, obsList )
        
        
    def test_getChainIdentityListFromSubject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "1", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "2", "qry1", "801", "900", "sbj1", "1", "100", "0.0", "187", "94.3" ) )
        self._tpA.insert( p3 )
        
        idChain1 = ( 98.5*(200-101+1) + 97.2*(500-351+1) ) / float(200-101+1+500-351+1)
        idChain2 = 94.3*(900-801+1) / float(900-801+1)
        expList = [ idChain1, idChain2 ]
        obsList = self._tpA.getChainIdentityListFromSubject( "sbj1" )
        self.assertEqual( expList, obsList )
        
        
    def test_getPathLengthListFromSubject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "1", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "2", "qry1", "801", "900", "sbj1", "1", "100", "0.0", "187", "97.2" ) )
        self._tpA.insert( p3 )
        
        expList = [ 100, 150, 100 ]
        obsList = self._tpA.getPathLengthListFromSubject( "sbj1" )
        self.assertEqual( expList, obsList )
        
        
    def test_getIdListSortedByDecreasingChainLengthFromSubject( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0847", "100", "97.2" ) )
        self._tpA.insert( p1 )
        p2a = Path()
        p2a.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "1", "150", "0.0035", "100", "97.2" ) )
        self._tpA.insert( p2a )
        p2b = Path()
        p2b.setFromTuple( ( "2", "qry1", "551", "700", "sbj1", "151", "300", "0.0098", "100", "97.2" ) )
        self._tpA.insert( p2b )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "541", "800", "sbj1", "1", "260", "0.147", "100", "97.2" ) )
        self._tpA.insert( p3 )
        
        expList = [ 2, 3, 1 ]
        obsList = self._tpA.getIdListSortedByDecreasingChainLengthFromSubject( "sbj1" )
        
        self.assertEqual( expList, obsList )
        
        
    def test_getIdListFromSubjectWhereChainsLongerThanThreshold( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "1", "100", "sbj1", "1", "100", "0.0847", "100", "97.2" ) )  # 1-fragment copy, long enough
        self._tpA.insert( p1 )
        p2a = Path()
        p2a.setFromTuple( ( "2", "qry2", "1", "50", "sbj1", "1", "50", "0.0035", "100", "97.2" ) )
        self._tpA.insert( p2a )
        p2b = Path()
        p2b.setFromTuple( ( "2", "qry2", "101", "150", "sbj1", "51", "100", "0.0098", "100", "97.2" ) )  # 2-fragment copy, long enough
        self._tpA.insert( p2b )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry3", "1", "80", "sbj1", "1", "80", "0.147", "100", "97.2" ) )  # 1-fragment copy, too short
        self._tpA.insert( p3 )
        p4a = Path()
        p4a.setFromTuple( ( "4", "qry4", "1", "30", "sbj1", "1", "30", "0.0035", "100", "97.2" ) )
        self._tpA.insert( p4a )
        p4b = Path()
        p4b.setFromTuple( ( "4", "qry4", "101", "150", "sbj1", "31", "80", "0.0098", "100", "97.2" ) )  # 2-fragment copy, too short
        self._tpA.insert( p4b )
        
        exp = [ 1, 2 ]
        obs = self._tpA.getIdListFromSubjectWhereChainsLongerThanThreshold( "sbj1", 90 )
        
        self.assertEqual( exp, obs )
        
        
    def test_getSetListFromQuery(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "1", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "2", "qry1", "801", "900", "sbj1", "1", "100", "0.0", "187", "97.2" ) )
        self._tpA.insert( p3 )
        
        set1 = Set()
        set1.setFromTuple( ( "1", "sbj1", "qry1", "101", "200" ) )
        set2 = Set()
        set2.setFromTuple( ( "1", "sbj1", "qry1", "351", "500" ) )    
        set3 = Set()
        set3.setFromTuple( ( "2", "sbj1", "qry1", "801", "900" ) )
        expList = [set1, set2, set3]
        obsList =  self._tpA.getSetListFromQuery("qry1")
        self.assertEqual( expList, obsList )
        
        
    def test_deleteFromId(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "1", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "2", "qry1", "801", "900", "sbj1", "1", "100", "0.0", "187", "97.2" ) )
        self._tpA.insert( p3 )
        
        self._tpA.deleteFromId(2)
        expList = [p1, p2]
        obsList = self._tpA.getListOfAllPaths()
        self.assertEqual( expList, obsList )
        
        
    def test_deleteFromPath(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "1", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "2", "qry1", "801", "900", "sbj1", "1", "100", "0.0", "187", "97.2" ) )
        self._tpA.insert( p3 )
        
        self._tpA.deleteFromPath(p3)
        expList = [p1, p2]
        obsList = self._tpA.getListOfAllPaths()
        self.assertEqual( expList, obsList )
        
    def test_deleteFromPath_two_lines_to_delete(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "1", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "2", "qry1", "801", "900", "sbj1", "1", "100", "0.0", "187", "97.2" ) )
        self._tpA.insert( p3 )
        p4 = Path()
        p4.setFromTuple( ( "2", "qry1", "801", "900", "sbj1", "1", "100", "0.0", "187", "97.2" ) )
        self._tpA.insert( p4 )
        
        self._tpA.deleteFromPath(p3)
        expList = [p1, p2]
        obsList = self._tpA.getListOfAllPaths()
        self.assertEqual( expList, obsList )
        
        
    def test_deleteFromIdList(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry1", "801", "900", "sbj1", "1", "100", "0.0", "187", "97.2" ) )
        self._tpA.insert( p3 )
        lId = [ 1, 2 ]
        self._tpA.deleteFromIdList(lId)
        expList = [ p3 ]
        obsList = self._tpA.getListOfAllPaths()
        self.assertEqual( expList, obsList )
        
        
    def test_joinTwoPaths(self):
        self._db.createTable( self._table, "path" )
        
        idPath1 = 5
        p1 = Path()
        p1.setFromTuple( ( "5", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        
        idPath2 = 2
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        
        lPath = [ p1, p2 ]
        self._tpA.insertList( lPath )

        self._tpA.joinTwoPaths(idPath1, idPath2)

        expP1 = Path()
        expP1.setFromTuple( ( "2", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        expP2 = Path()
        expP2.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        
        expList = [ expP1, expP2 ]
        obsList = self._tpA.getListOfAllPaths()
        
        self.assertEqual( expList, obsList)
        self._db.dropTable(self._table)        
        
        
    def test_joinTwoPaths_with_id1_inferior_at_id2(self):
        self._db.createTable( self._table, "path" )
        
        idPath1 = 5
        p1 = Path()
        p1.setFromTuple( ( "5", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        
        idPath2 = 2
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        
        lPath = [ p1, p2 ]
        self._tpA.insertList( lPath )

        self._tpA.joinTwoPaths(idPath2, idPath1)

        expP1 = Path()
        expP1.setFromTuple( ( "2", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        expP2 = Path()
        expP2.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        
        expList = [ expP1, expP2 ]
        obsList = self._tpA.getListOfAllPaths()
        
        self.assertEqual( expList, obsList)
        self._db.dropTable(self._table)       
        
        
    def test_getNewId(self):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0", "137", "98.5" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj1", "101", "250", "0.0", "187", "97.2" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry1", "801", "900", "sbj1", "1", "100", "0.0", "187", "97.2" ) )
        self._tpA.insert( p3 )
        
        expId = 4
        obsId = self._tpA.getNewId()
        self.assertEqual( expId, obsId)
        
        
    def test_getNewId_path_null(self):
        self._db.createTable( self._table, "path" )
        expId = 1
        obsId = self._tpA.getNewId()
        self.assertEqual( expId, obsId)
        
        
    def test_getListOfChainsSortedByAscIdentityFromQuery( self ):
        self._db.createTable( self._table, "path" )
        p1a = Path()
        p1a.setFromTuple( ( "1", "qry1", "11", "100", "sbj1", "1", "90", "0.0", "132", "96.2" ) )
        p2a = Path()
        p2a.setFromTuple( ( "2", "qry1", "101", "120", "sbj2", "1", "20", "0.0", "36", "98.0" ) )
        p1b = Path()
        p1b.setFromTuple( ( "1", "qry1", "121", "200", "sbj1", "91", "170", "0.0", "117", "96.5" ) )
        p2b = Path()
        p2b.setFromTuple( ( "2", "qry1", "201", "800", "sbj2", "21", "620", "0.0", "856", "93.2" ) )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "1", "1000", "sbj1", "1", "1000", "1e-120", "900", "100.0" ) )
        for p in [ p1a, p2a, p1b, p2b, p3 ]: self._tpA.insert( p )
        lPaths1 = [ p1a, p1b ]
        lPaths2 = [ p2a, p2b ]
        expList = [ lPaths2, lPaths1 ]
        obsList = self._tpA.getListOfChainsSortedByAscIdentityFromQuery( "qry1" )
        for lPaths in expList: PathUtils.getPathListSortedByIncreasingMinQueryThenMaxQuery( lPaths )
        for lPaths in obsList: PathUtils.getPathListSortedByIncreasingMinQueryThenMaxQuery( lPaths )
        self.assertEqual( expList, obsList )
        
        
    def test_getPathListSortedByIncreasingEvalueFromQuery( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "qry1", "101", "200", "sbj1", "1", "100", "0.0847", "100", "97.2" ) )
        self._tpA.insert( p1 )
        p2 = Path()
        p2.setFromTuple( ( "2", "qry1", "351", "500", "sbj2", "1", "150", "0.0035", "100", "97.2" ) )
        self._tpA.insert( p2 )
        p3 = Path()
        p3.setFromTuple( ( "3", "qry2", "541", "800", "sbj3", "1", "260", "0.147", "100", "97.2" ) )
        self._tpA.insert( p3 )
        
        expList = [ p2, p1 ]
        obsList = self._tpA.getPathListSortedByIncreasingEvalueFromQuery( "qry1" )
        self.assertEqual( expList, obsList )
        
        
    def test_path2PathRange_QryDirSbjDir( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "chr1", "1", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        p2a = Path()
        p2a.setFromTuple( ( "2", "chr2", "1", "100", "TE2", "10", "109", "1e-20", "163", "92.1" ) )
        p2b = Path()
        p2b.setFromTuple( ( "2", "chr2", "201", "250", "TE2", "151", "200", "1e-10", "75", "88.7" ) )
        for p in [ p1, p2a, p2b ]: self._tpA.insert( p )
        p2 = Path()
        
        p2.setFromTuple( ( "2", "chr2", "1", "250", "TE2", "10", "200", "1e-20", "238", "90.96" ) )   # 'merge' p2a and p2b
        expList = [ p1, p2 ]
        obsTable = self._tpA.path2PathRange()
        self._tpA._table = obsTable
        obsList = self._tpA.getListOfAllPaths()
        self.assertEqual( expList, obsList )
        self._db.dropTable( obsTable )
        
        
    def test_path2PathRange_QryDirSbjRev( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "chr1", "1", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        p2a = Path()
        p2a.setFromTuple( ( "2", "chr2", "1", "100", "TE2", "109", "10", "1e-20", "163", "92.1" ) )
        p2b = Path()
        p2b.setFromTuple( ( "2", "chr2", "201", "250", "TE2", "200", "151", "1e-10", "75", "88.7" ) )
        for p in [ p1, p2a, p2b ]: self._tpA.insert( p )
        p2 = Path()
        p2.setFromTuple( ( "2", "chr2", "1", "250", "TE2", "200", "10", "1e-20", "238", "90.96" ) )   # 'merge' p2a and p2b
        expList = [ p1, p2 ]
        obsTable = self._tpA.path2PathRange()
        self._tpA._table = obsTable
        obsList = self._tpA.getListOfAllPaths()
        self.assertEqual( obsList, expList )
        self._db.dropTable( obsTable )
        
        
###################################################################################
############################ Tests for other methods ##############################
###################################################################################
        
    def test_path2PathRangeFromQuery_QryDirSbjDir( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "chr1", "1", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        p2a = Path()
        p2a.setFromTuple( ( "2", "chr2", "1", "100", "TE2", "10", "109", "1e-20", "163", "92.1" ) )
        p2b = Path()
        p2b.setFromTuple( ( "2", "chr2", "201", "250", "TE2", "151", "200", "1e-10", "75", "88.7" ) )
        for p in [ p1, p2a, p2b ]: self._tpA.insert( p )
        p2 = Path()
        p2.setFromTuple( ( "2", "chr2", "1", "250", "TE2", "10", "200", "1e-20", "238", "90.96" ) )   # 'merge' p2a and p2b
        expList = [ p2 ]
        obsTable = self._tpA._path2PathRangeFromQuery( "chr2" )
        self._tpA._table = obsTable
        obsList = self._tpA.getListOfAllPaths()
        self.assertEqual( expList, obsList )
        self._db.dropTable( obsTable )
        
        
    def test_path2PathRangeFromQuery_QryDirSbjRev( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "chr1", "1", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        p2a = Path()
        p2a.setFromTuple( ( "2", "chr2", "1", "100", "TE2", "109", "10", "1e-20", "163", "92.1" ) )
        p2b = Path()
        p2b.setFromTuple( ( "2", "chr2", "201", "250", "TE2", "200", "151", "1e-10", "75", "88.7" ) )
        for p in [ p1, p2a, p2b ]: self._tpA.insert( p )
        p2 = Path()
        p2.setFromTuple( ( "2", "chr2", "1", "250", "TE2", "200", "10", "1e-20", "238", "90.96" ) )   # 'merge' p2a and p2b
        expList = [ p2 ]
        obsTable = self._tpA._path2PathRangeFromQuery( "chr2" )
        self._tpA._table = obsTable
        obsList = self._tpA.getListOfAllPaths()
        self.assertEqual( obsList, expList )
        self._db.dropTable( obsTable )
        
        
    def test_getNbOccurrences( self ):
        self._db.createTable( self._table, "path" )
        p1 = Path()
        p1.setFromTuple( ( "1", "chr1", "1", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        
        exp = 0
        obs = self._tpA.getNbOccurrences( p1 )
        self.assertEquals( exp, obs )
        
        self._tpA.insert( p1 )
        exp = 1
        obs = self._tpA.getNbOccurrences( p1 )
        self.assertEquals( exp, obs )
        
        self._tpA.insert( p1 )
        exp = 2
        obs = self._tpA.getNbOccurrences( p1 )
        self.assertEquals( exp, obs )
        
    def test_getListOfUniqueOccPath(self):
        
        p1 = Path()
        p1.setFromTuple( ( "1", "chr1", "1", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        p2 = Path()
        p2.setFromTuple( ( "1", "chr1", "1", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        p3 = Path()
        p3.setFromTuple( ( "1", "chr1", "2", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        p4 = Path()
        p4.setFromTuple( ( "2", "chr2", "2", "11", "TE4", "10", "18", "1e-30", "40", "95.0" ) )
        lPath = [p1,p2,p3,p4]
                
        expListPath = deepcopy([p1,p3,p4])     
        obsListUniquePath = self._tpA.getListOfUniqueOccPath(lPath)
        self.assertEquals( expListPath, obsListUniquePath )

    def test_getListOfUniqueOccPath_empty_list(self):
        expListPath = []     
        obsListUniquePath = self._tpA.getListOfUniqueOccPath([])
        self.assertEquals( expListPath, obsListUniquePath )
        
    def test_getListOfUniqueOccPath_one_item(self):
        p1 = Path()
        p1.setFromTuple( ( "1", "chr1", "1", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        expListPath = deepcopy([p1])      
        obsListUniquePath = self._tpA.getListOfUniqueOccPath([p1])
        self.assertEquals( expListPath, obsListUniquePath )

    def test_getListOfUniqueOccPath_unsorted_list(self):
        
        p1 = Path()
        p1.setFromTuple( ( "1", "chr1", "1", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        p3 = Path()
        p3.setFromTuple( ( "1", "chr1", "3", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        p4 = Path()
        p4.setFromTuple( ( "2", "chr2", "2", "11", "TE4", "10", "18", "1e-30", "40", "95.0" ) )
        p2 = Path()
        p2.setFromTuple( ( "1", "chr1", "1", "10", "TE3", "11", "17", "1e-20", "30", "85.0" ) )
        
        lPath = [p1,p3,p4,p2]
                
        expListPath = deepcopy([p1,p3,p4])     
        obsListUniquePath = self._tpA.getListOfUniqueOccPath(lPath)
        self.assertEquals( expListPath, obsListUniquePath )

test_suite = unittest.TestSuite()
test_suite.addTest( unittest.makeSuite( Test_TablePathAdaptator ) )
if __name__ == "__main__":
    unittest.TextTestRunner(verbosity=2).run( test_suite )
