import random
from mySql.mySqlTranscriptTable import *
from misc.progress import *

class MySqlTranscriptWriter(object):
  """
  A class that writes a transcript list into a mySQL table
  @ivar name:                      name of the tables 
  @type name:                      string
  @ivar tables:                    the tables
  @type tables:                    dict of L{MySqlTranscriptTable<MySqlTranscriptTable>}
  @ivar mySqlConnection:           connection to a MySQL database
  @type mySqlConnection:           class L{MySqlConnection<MySqlConnection>}
  @ivar tmpTranscriptFileHandles:  files where transcripts are temporary stored, before copy into database
  @type tmpTranscriptFileHandles:  dict of file handles
  @ivar tmpExonFileHandles:        files where exons are temporary stored, before copy into database
  @type tmpExonFileHandles:        dict of file handles
  @ivar nbTranscriptsByChromosome: number of transcripts written
  @type nbTranscriptsByChromosome: dict of int (one for each chromosome)
  @ivar nbExonsByChromosome:       number of exons written
  @type nbExonsByChromosome:       dict of int (one for each chromosome)
  @ivar randomNumber:              a random number, used for having a unique name for the tables
  @type randomNumber:              int
  @ivar toBeWritten:               there exists transcripts to be copied into database
  @type toBeWritten:               bool        
  @ivar verbosity:                 verbosity
  @type verbosity:                 int    
  """


  def __init__(self, name = None, verbosity = 0):
    """
    Constructor
    @param name:      name of the file 
    @type name:       string
    @param verbosity: verbosity
    @type verbosity:  int
    """
    self.name                      = name
    self.verbosity                 = verbosity
    self.tables                    = {}
    self.tmpTranscriptFileHandles  = {}
    self.tmpExonFileHandles        = {}
    self.nbTranscriptsByChromosome = {}
    self.nbExonsByChromosome       = {}
    self.toBeWritten               = False
    self.randomNumber              = random.randint(0, 100000)
    self.mySqlConnection           = MySqlConnection(self.verbosity)
    self.nbTmpFiles                = 100
    if self.name != None:
      pos = self.name.rfind(os.sep)
      if pos != -1:
        self.name = self.name[pos+1:]
      

  def __del__(self):
    """
    Destructor
    Possibly write into into database the last transcripts
    """
    if self.toBeWritten:
      self.write()
  
  
  def addTranscript(self, transcript):
    """
    Add a transcript to the list of transcripts to be written
    @param transcript: transcript to be written
    @type  transcript: class L{Transcript<Transcript>}
    """
    chromosome = transcript.chromosome
    if chromosome not in self.tmpTranscriptFileHandles:
      self.tmpTranscriptFileHandles[chromosome]  = open("tmpTranscriptFile_%s_%d.dat" % (chromosome, self.randomNumber), "w")
      self.tmpExonFileHandles[chromosome]        = open("tmpExonFile_%s_%d.dat" % (chromosome, self.randomNumber), "w")
    if chromosome not in self.nbTranscriptsByChromosome:
      self.nbTranscriptsByChromosome[chromosome] = 1
      self.nbExonsByChromosome[chromosome]       = 1
      
    # write transcript values
    values = transcript.getSqlValues()
    types  = transcript.getSqlTypes()
    sizes  = transcript.getSqlSizes()    
    self.tmpTranscriptFileHandles[chromosome].write("%d" % (self.nbTranscriptsByChromosome[chromosome]))
    for variable in transcript.getSqlVariables():
      self.tmpTranscriptFileHandles[chromosome].write("\t%s" % MySqlTable.formatSql(values[variable], types[variable], sizes[variable]).replace("'", ""))
    self.tmpTranscriptFileHandles[chromosome].write("\n")
    
    for exon in transcript.getExons():
      # write exon values
      values = exon.getSqlValues()
      types  = exon.getSqlTypes()
      sizes  = exon.getSqlSizes()
      self.tmpExonFileHandles[chromosome].write("%d" % (self.nbExonsByChromosome[transcript.chromosome]))
      for variable in exon.getSqlVariables():
        self.tmpExonFileHandles[chromosome].write("\t%s" % (MySqlTable.formatSql(values[variable], types[variable], sizes[variable]).replace("'", "")))
      self.tmpExonFileHandles[chromosome].write("\t%d\n" % (self.nbTranscriptsByChromosome[transcript.chromosome]))
      self.nbExonsByChromosome[chromosome] += 1

    self.nbTranscriptsByChromosome[chromosome] += 1
    self.toBeWritten                            = True
    if len(self.tmpTranscriptFileHandles.keys()) > self.nbTmpFiles:
      self.write() 


  def addElement(self, element):
    """
    Same as "addTranscript"
    @param element: transcript to be written
    @type  element: class L{Transcript<Transcript>}
    """
    self.addTranscript(element)


  def addTranscriptList(self, transcriptListParser):
    """
    Add a list of transcripts to the transcripts to be written
    @param transcriptListParser: transcripts to be written
    @type  transcriptListParser: class L{TranscriptListParser<TranscriptListParser>}
    """
    progress = Progress(transcriptListParser.getNbTranscripts(), "Storing %s into database" % (transcriptListParser.fileName), self.verbosity)
    for transcript in transcriptListParser.getIterator():
      self.addTranscript(transcript)
      progress.inc()
    progress.done()
      
      
  def write(self):
    """
    Copy the content of the files into the database
    (May add transcripts to already created databases)
    """
    for chromosome in self.tmpTranscriptFileHandles:
      self.tmpTranscriptFileHandles[chromosome].close()
      self.tmpExonFileHandles[chromosome].close()
      if chromosome not in self.tables:
        self.tables[chromosome] = MySqlTranscriptTable(self.name, chromosome, self.verbosity)
        self.tables[chromosome].createTranscriptTable()
      self.mySqlConnection.executeQuery("LOAD DATA LOCAL INFILE '%s%stmpTranscriptFile_%s_%d.dat' INTO TABLE %s" % (os.getcwd(), os.sep, chromosome, self.randomNumber, self.tables[chromosome].name))
      self.mySqlConnection.executeQuery("LOAD DATA LOCAL INFILE '%s%stmpExonFile_%s_%d.dat' INTO TABLE %s" % (os.getcwd(), os.sep, chromosome, self.randomNumber, self.tables[chromosome].exonsTable.name))
      self.mySqlConnection.executeQuery("ALTER TABLE %s ORDER BY bin" % (self.tables[chromosome].name))
      os.remove("tmpTranscriptFile_%s_%d.dat" % (chromosome, self.randomNumber))
      os.remove("tmpExonFile_%s_%d.dat" % (chromosome, self.randomNumber))
    self.tmpTranscriptFileHandles = {}
    self.tmpExonFileHandles       = {}
    self.toBeWritten              = False
      
      
  def getTables(self):
    """
    Get the tables
    @return: the mySQL tables
    """
    if self.toBeWritten:
      self.write()
    return self.tables
