import re
import sys
from structure.interval import *


class SubMapping(object):
  """
  A class that represents a part of a mapping, more precisely, a pair (target interval, query interval) that match together
  @ivar targetInterval: the target interval
  @type targetInterval: class L{Interval<Interval>}
  @ivar queryInterval:  the query interval
  @type queryInterval:  class L{Interval<Interval>}
  @ivar size:           size of this sub-mapping
  @type size:           int
  @ivar direction:      whether target and query intervals are in the same directions
  @type direction:      int (1 / -1)
  @ivar tags:           various information
  @type tags:           dict
  """

  def __init__(self, subMapping = None):
    """
    Constructor
    @param subMapping: a sub-mapping to be copied
    @type  subMapping: class L{SubMapping<SubMapping>}
    """
    self.targetInterval = Interval()
    self.queryInterval  = Interval()
    self.size           = None
    self.direction      = 0
    self.tags           = {}
    if subMapping != None:
      self.copy(subMapping)


  def copy(self, subMapping):
    """
    Copy method
    @param subMapping: a sub-mapping to be copied
    @type  subMapping: class L{SubMapping<SubMapping>}
    """
    self.targetInterval.copy(subMapping.targetInterval)
    self.queryInterval.copy(subMapping.queryInterval)
    self.size      = subMapping.size
    self.direction = subMapping.direction
    for tag in subMapping.tags:
      self.tags[tag] = subMapping.tags[tag]


  def setTargetInterval(self, interval):
    """
    Set target interval
    @param targetInterval: the target interval of the sub-mapping
    @type  targetInterval: class L{Interval<Interval>}
    """
    self.targetInterval.copy(interval)


  def setQueryInterval(self, interval):
    """
    Set query interval
    @param queryInterval: the query interval of the sub-mapping
    @type  queryInterval: class L{Interval<Interval>}
    """
    self.queryInterval.copy(interval)


  def setSize(self, size):
    """
    Set the size of the sub-mapping
    Possibly also target and query interval sizes, as well as number of mismatches
    @param size: the size of the sub-mapping
    @type  size: int
    """
    self.size = size
    if self.queryInterval.size == None:
      self.queryInterval.size = self.size
    if self.targetInterval.size == None:
      self.targetInterval.size = self.size
    if "identity" in self.getTagNames():
      self.setTagValue("nbMismatches", self.size - round(self.size * self.getTagValue("identity") / 100.0))


  def setDirection(self, direction):
    """
    Set the direction of the sub-mapping
    @param direction: the direction of the sub-mapping
    @type  direction: int or string
    """
    if type(direction).__name__ == 'int':
      self.direction = direction / abs(direction)
    elif type(direction).__name__ == 'str':
      if direction == "+":
        self.direction = 1
      elif direction == "-":
        self.direction = -1
      elif direction == "1" or direction == "-1":
        self.direction = int(direction)
      else:
        sys.exit("Cannot understand direction " + direction)
    else:
      sys.exit("Cannot understand direction " + direction)


  def setTagValue(self, name, value):
    """
    Set the value of a tag
    @param name:  name of the tag
    @type  name:  string
    @param value: value of the tag
    @type  value: string or int
    """
    self.tags[name] = value


  def getTagValue(self, name):
    """
    Get the value of a tag
    @param name:  name of the tag
    @type  name:  string
    @return:      value of the tag
    """
    return self.tags[name]

  
  def getTagNames(self):
    """
    Get all the names of the tags
    @return: the names of the tags
    """
    return self.tags.keys()


  def setIdentity(self, identity):
    """
    Set the percentage of identity of the sub-mapping
    Possibly also set number of mismatches
    @param identity: the percentage of identity of the sub-mapping
    @type  identity: float
    """
    self.setTagValue("identity", identity)
    if self.size != None and "nbMismatches" not in self.getTagNames():
      self.setTagValue("nbMismatches", self.size - round(self.size * self.getTagValue("identity") / 100.0))


  def setNbMismatches(self, nbMismatches):
    """
    Set the number of mismatches of the sub-mapping
    Possibly also set percentage of identity
    @param nbMismatches: the number of mismatches of the sub-mapping
    @type  nbMismatches: int
    """
    self.nbMismatches = nbMismatches
    if self.size != None and "identity" not in self.getTagNames():
      self.setTagValue("identity", (self.size - self.getTagValue("nbMismatches")) / float(self.size) * 100)


  def setNbGaps(self, nbGaps):
    """
    Set the number of gaps of the sub-mapping
    @param nbGaps: the number of gaps of the sub-mapping
    @type  nbGaps: int
    """
    self.setTagValue("nbGaps", nbGaps)
    
    
  def merge(self, subMapping):
    """
    Merge two subMappings
    @param subMapping: another sub-mapping
    @type  subMapping: class L{SubMapping<SubMapping>}
    """
    if self.targetInterval.chromosome != subMapping.targetInterval.chromosome or self.queryInterval.chromosome != subMapping.queryInterval.chromosome or self.targetInterval.direction != subMapping.targetInterval.direction or self.queryInterval.direction != subMapping.queryInterval.direction:
      sys.exit("Cannot merge sub-mappings %s and %s" % (str(self), str(subMapping)))
    self.targetInterval.merge(subMapping.targetInterval)
    self.queryInterval.merge(subMapping.queryInterval)


  def printCoordinates(self):
    """
    Print the coordinates of the sub-mapping (considering the direction)
    @return: a string
    """
    if self.direction == 1:
      return "%d-%d" % (self.targetInterval.start, self.targetInterval.end)
    else:
      return "%d-%d" % (self.targetInterval.end, self.targetInterval.start)


  def __str__(self):
    """
    Return a representation of this object
    @return: a string
    """

    if "match" in self.getTagNames() and not self.getTagValue("match"):
      return "%s ---" % self.queryName

    direction = "+"
    if self.direction == -1:
      direction = "-"
    string = "%s:%d-%d -- %s:%d-%d  (%s)" % (self.targetInterval.chromosome, self.targetInterval.start, self.targetInterval.end, self.queryInterval.chromosome, self.queryInterval.start, self.queryInterval.end, direction)
    if "nbMismatches" in self.getTagNames():
      string += "(%i mm)" % (self.getTagValue("nbMismatches"))
    if "identity" in self.getTagNames():
      string += "(id: %i%%)" % (self.getTagValue("identity"))
    if self.targetInterval.size != None and self.queryInterval.size != None and self.size != None:
      string += "(sizes: %d, %d -> %d)" % (self.targetInterval.size, self.queryInterval.size, self.size)
    return string


