import re
import sys
from structure.mapping import *


class MapperParser(object):
  """An interface that parses the output of a generic mapper"""

  def __init__(self, fileName, verbosity = 0):
    super(MapperParser, self).__init__()
    self.verbosity      = verbosity
    self.nbMappings     = None
    self.chromosomes    = None
    self.size           = None
    self.currentMapping = Mapping()
    self.handle         = open(fileName)
    self.currentLineNb  = 0
    self.skipFirstLines()
    self.startingPoint = self.handle.tell()


  def __del__(self):
    self.handle.close()
    

  def reset(self):
    self.handle.seek(self.startingPoint)
    self.currentLineNb = 0

    
  def getIterator(self):
    self.reset()
    for line in self.handle:
      mapping = self.parseLine(line)
      self.currentLineNb += 1
      if mapping != None:
        yield mapping
        
        
  def getInfos(self):
    self.chromosomes = set()
    self.nbMappings  = 0
    self.size        = 0
    self.reset()
    if self.verbosity >= 10:
      print "Getting information."
    for mapping in self.getIterator():
      transcript = mapping.getTranscript()
      self.chromosomes.add(transcript.chromosome)
      self.nbMappings += 1
      self.size       += transcript.getSize()
      if self.verbosity >= 10 and self.nbMappings % 100000 == 0:
        sys.stdout.write("  %d mappings read\r" % (self.nbMappings))
        sys.stdout.flush()
    self.reset()
    if self.verbosity >= 10:
      print "  %d mappings read" % (self.nbMappings)
      print "Done."


  def getNbMappings(self):
    if self.nbMappings != None:
      return self.nbMappings
    self.getInfos()
    return self.nbMappings


  def getNbItems(self):
    return self.getNbMappings()


  def getChromosomes(self):
    if self.chromosomes != None:
      return self.chromosomes
    self.getInfos()
    return self.chromosomes
  
  
  def getSize(self):
    if self.size != None:
      return self.size
    self.getInfos()
    return self.size
  
  
  def getNbNucleotides(self):
    return self.getSize()


  def setDefaultTagValue(self, name, value):
    for mapping in self.getIterator():
      mapping.setTagValue(name, value)
