#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
from SMART.Java.Python.structure.Interval import Interval
from commons.core.coord.Align import Align

class SubMapping(Align):
    """
    A class that represents a part of a mapping, more precisely, a pair (target interval, query interval) that match together
    @ivar targetInterval: the target interval
    @type targetInterval: class L{Interval<Interval>}
    @ivar queryInterval:  the query interval
    @type queryInterval:  class L{Interval<Interval>}
    @ivar size:           size of this sub-mapping
    @type size:           int
    @ivar tags:           various information
    @type tags:           dict
    """

    def __init__(self, subMapping = None):
        """
        Constructor
        @param subMapping: a sub-mapping to be copied
        @type  subMapping: class L{SubMapping<SubMapping>}
        """
        self.targetInterval = Interval()
        self.queryInterval  = Interval()
        Align.__init__(self, self.queryInterval, self.targetInterval)
        self.size = None
        self.tags = {}
        if subMapping != None:
            self.copy(subMapping)
    
    def __eq__(self, o):
        if o == None:
            return False
        areAlignAttributesEquals = Align.__eq__(self, o)
        return areAlignAttributesEquals and (self.targetInterval == o.targetInterval) and (self.queryInterval == o.queryInterval) and self.size == o.getSize() and self.tags == o.getTags()
    
    def getSuperAdress(self):
        return hex(id(super(Align, self)))
    
#    def setRangesAlignToRangesInterval(self):
#        self.range_query = super(Range, self.queryInterval)
#        self.range_subject = super(Range, self.targetInterval)
        
    def copy(self, subMapping):
        """
        Copy method
        @param subMapping: a sub-mapping to be copied
        @type    subMapping: class L{SubMapping<SubMapping>}
        """
        self.setQueryName(subMapping.getQueryName())
        self.setQueryStart(subMapping.getQueryStart())
        self.setQueryEnd(subMapping.getQueryEnd())
        self.setSubjectName(subMapping.getSubjectName())
        self.setSubjectStart(subMapping.getSubjectStart())
        self.setSubjectEnd(subMapping.getSubjectEnd())
        self.e_value = subMapping.getEvalue()
        self.score = subMapping.getScore()
        self.identity = subMapping.getIdentity()
        
        self.targetInterval.copy(subMapping.targetInterval)
        self.queryInterval.copy(subMapping.queryInterval)
        self.size = subMapping.size
        for tag in subMapping.tags:
            self.tags[tag] = subMapping.tags[tag]


    def setTargetInterval(self, interval):
        """
        Set target interval
        @param targetInterval: the target interval of the sub-mapping
        @type    targetInterval: class L{Interval<Interval>}
        """
        self.targetInterval.copy(interval)


    def setQueryInterval(self, interval):
        """
        Set query interval
        @param queryInterval: the query interval of the sub-mapping
        @type    queryInterval: class L{Interval<Interval>}
        """
        self.queryInterval.copy(interval)


    def setSize(self, size):
        """
        Set the size of the sub-mapping
        Possibly also target and query interval sizes, as well as number of mismatches
        @param size: the size of the sub-mapping
        @type    size: int
        """
        self.size = size
        if "identity" in self.getTagNames():
            self.setTagValue("nbMismatches", self.size - round(self.size * self.getTagValue("identity") / 100.0))


    def getDirection(self):
        """
        Get the direction of the alignment
        """
        return self.targetInterval.getDirection()


    def setDirection(self, direction):
        """
        Set the direction of the alignment
        @param direction: the directio of the alignment
        type   direction: int or string
        """
        return self.targetInterval.setDirection(direction)


    def setTagValue(self, name, value):
        """
        Set the value of a tag
        @param name:    name of the tag
        @type    name:    string
        @param value: value of the tag
        @type    value: string or int
        """
        self.tags[name] = value


    def getTagValue(self, name):
        """
        Get the value of a tag
        @param name:    name of the tag
        @type    name:    string
        @return:            value of the tag
        """
        return self.tags[name]

    
    def getTagNames(self):
        """
        Get all the names of the tags
        @return: the names of the tags
        """
        return self.tags.keys()

    def getTargetInterval(self):
        return self.targetInterval
    
    def getQueryInterval(self):
        return self.queryInterval
    
    def getSize(self):
        return self.size
    
    def getTags(self):
        return self.tags

    def setIdentity(self, identity):
        """
        Set the percentage of identity of the sub-mapping
        Possibly also set number of mismatches
        @param identity: the percentage of identity of the sub-mapping
        @type  identity: float
        """
        self.identity = identity
        self.setTagValue("identity", identity)
        if self.size != None and "nbMismatches" not in self.getTagNames():
            self.setTagValue("nbMismatches", self.size - round(self.size * self.getTagValue("identity") / 100.0))


    def setNbMismatches(self, nbMismatches):
        """
        Set the number of mismatches of the sub-mapping
        Possibly also set percentage of identity
        @param nbMismatches: the number of mismatches of the sub-mapping
        @type    nbMismatches: int
        """
        self.nbMismatches = nbMismatches
        if self.size != None and "identity" not in self.getTagNames():
            self.setTagValue("identity", (self.size - self.getTagValue("nbMismatches")) / float(self.size) * 100)


    def setNbGaps(self, nbGaps):
        """
        Set the number of gaps of the sub-mapping
        @param nbGaps: the number of gaps of the sub-mapping
        @type    nbGaps: int
        """
        self.setTagValue("nbGaps", nbGaps)
        
        
    def merge(self, subMapping):
        """
        Merge two subMappings
        @param subMapping: another sub-mapping
        @type    subMapping: class L{SubMapping<SubMapping>}
        """
        self.targetInterval.merge(subMapping.targetInterval)
        self.queryInterval.merge(subMapping.queryInterval)


    def printCoordinates(self):
        """
        Print the coordinates of the sub-mapping (considering the direction)
        @return: a string
        """
        if self.getDirection() == 1:
            return "%d-%d" % (self.targetInterval.getStart(), self.targetInterval.getEnd())
        else:
            return "%d-%d" % (self.targetInterval.getEnd(), self.targetInterval.getStart())


    def __str__(self):
        """
        Return a representation of this object
        @return: a string
        """

        if "match" in self.getTagNames() and not self.getTagValue("match"):
            return "%s ---" % self.queryName

        direction = "+"
        if self.getDirection() == -1:
            direction = "-"
        string = "%s:%d-%d -- %s:%d-%d    (%s)" % (self.targetInterval.getChromosome(), self.targetInterval.getStart(), self.targetInterval.getEnd(), self.queryInterval.name, self.queryInterval.getStart(), self.queryInterval.getEnd(), direction)
        if "nbMismatches" in self.getTagNames():
            string += "(%i mm)" % (self.getTagValue("nbMismatches"))
        if "identity" in self.getTagNames():
            string += "(id: %i%%)" % (self.getTagValue("identity"))
        if self.targetInterval.getSize() != None and self.queryInterval.getSize() != None and self.size != None:
            string += "(sizes: %d, %d -> %d)" % (self.targetInterval.getSize(), self.queryInterval.getSize(), self.size)
        return string


