#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
from optparse import OptionParser
from commons.core.parsing.ParserChooser import ParserChooser
from commons.core.writer.TranscriptWriter import TranscriptWriter
from SMART.Java.Python.structure.Transcript import Transcript
from SMART.Java.Python.misc.Progress import Progress

zeroBaseToOneBaseConvertor = (lambda x: x - 1 if x > 0 else x)

class GetExons(object):

    def __init__(self, verbosity):
        self.verbosity = verbosity
        self.selection = False

    def setInputFile(self, fileName, format):
        chooser = ParserChooser(self.verbosity)
        chooser.findFormat(format)
        self.parser = chooser.getParser(fileName)

    def setSelection(self, selection):
        if selection == None:
            return
        self.selection = True
        self.selectionItems = []
        self.selectionIntervals = []
        for part in selection.split(","):
            try:
                splittedPart = map(int, part.split(".."))
            except Exception:
                raise Exception("Elements '" + splittedPart + "' of selection '" + selection + "' do no seem to be integers!")
            if len(splittedPart) == 1:
                self.selectionItems.append(splittedPart[0])
            elif len(splittedPart) == 2:
                self.selectionIntervals.append((splittedPart[0], splittedPart[1]))
            else:
                raise Exception("Cannot parse elements '" + splittedPart + "' of selection '" + selection + "'!")

    def getSelectionExonIndices(self, nbExons):
        if not self.selection:
            return range(nbExons)
        indices = []
        for item in self.selectionItems:
            indices.append(range(nbExons)[zeroBaseToOneBaseConvertor(item)])
        for start, end in self.selectionIntervals:
            start, end = map(zeroBaseToOneBaseConvertor, (start, end))
            if end > 0:
                end += 1
            indices.extend(range(nbExons)[start:end])
        return indices

    def setOutputFile(self, fileName):
        self.writer = TranscriptWriter(fileName, "gff3", self.verbosity)
        
    def run(self):
        progress = Progress(self.parser.getNbTranscripts(), "Reading input file", self.verbosity)
        nbExons = 0
        for cpt1, transcript in enumerate(self.parser.getIterator()):
            selectedExons = self.getSelectionExonIndices(transcript.getNbExons())
            transcript.sortExons()
            for cpt2, exon in enumerate(transcript.getExons()):
                if cpt2 not in selectedExons:
                    continue
                exonTranscript = Transcript()
                exonTranscript.copy(exon)
                if "Parent" in exonTranscript.tags:
                    del exonTranscript.tags["Parent"]
                exonTranscript.tags["feature"] = "transcript"
                if "ID" not in exonTranscript.tags or exonTranscript.tags["ID"] == "unnamed transcript":
                    exonTranscript.tags["ID"] = "exon_%d-%d" % (cpt1+1, cpt2+1)
                if exonTranscript.getName() == "unnamed transcript":
                    exonTranscript.setName("exon_%d-%d" % (cpt1+1, cpt2+1))
                self.writer.addTranscript(exonTranscript)
                nbExons += 1
            progress.inc()
        self.writer.write()
        self.writer.close()
        progress.done()
        if self.verbosity > 1:
            print "%d transcripts read" % (self.parser.getNbTranscripts())
            print "%d exons written" % (nbExons)

if __name__ == "__main__":
    
    description = "Get Exons v1.0.1: Get the exons of a set of transcripts. [Category: Data Modification]"

    parser = OptionParser(description = description)
    parser.add_option("-i", "--input",     dest="inputFileName",  action="store",               type="string", help="input file [compulsory] [format: file in transcript format given by -f]")
    parser.add_option("-f", "--format",    dest="format",         action="store",               type="string", help="format of file [compulsory] [format: transcript file format]")
    parser.add_option("-s", "--select",    dest="select",         action="store", default=None, type="string", help="select some of the exons (like '1,2,5..-3,-1') [format: string]")
    parser.add_option("-o", "--output",    dest="outputFileName", action="store",               type="string", help="output file [format: output file in GFF3 format]")
    parser.add_option("-v", "--verbosity", dest="verbosity",      action="store", default=1,    type="int",    help="trace level [format: int]")
    (options, args) = parser.parse_args()

    ge = GetExons(options.verbosity)
    ge.setInputFile(options.inputFileName, options.format)
    ge.setSelection(options.select)
    ge.setOutputFile(options.outputFileName)
    ge.run()
