#
# Copyright INRA-URGI 2009-2012
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#

import os
import subprocess
import random
import math
from SMART.Java.Python.misc.RPlotter import RPlotter

NBCOLORS = 9

"""
Plot multiple curves with RPlotter
"""

class MultipleRPlotter(object):
	"""
	Plot some curves
	@ivar fileName: name of the file
	@type fileName: string
	@ivar height: height of the file
	@type height: int
	@ivar width: width of the file
	@type width: int
	@ivar plots: plots to be included
	@type plots: list of L{RPlotter{RPlotter}}
	@ivar keep: keep script lines
	@type keep: boolean
	@ivar format: format of the file
	@type format: string
	"""

	def __init__(self, fileName, verbosity = 0, keep = False):
		"""
		Constructor
		@param fileName: name of the file to produce
		@type  fileName: string
		@param verbosity: verbosity
		@type  verbosity: int
		@param keep: keep temporary files
		@type  keep: boolean
		"""
		self.fileName = fileName
		self.verbosity = verbosity
		self.keep = keep
		self.format	= "png"
		self.width = 1000
		self.height	= 500
		self.plots = []
		self.scriptFileName = "tmpScript-%d.R" % (os.getpid())
	
	def __del__(self):
		"""
		Destructor
		Remove script files
		"""
		if not self.keep:
			if os.path.exists(self.scriptFileName):
				os.remove(self.scriptFileName)
			outputFileName = "%sout" % (self.scriptFileName)
			if os.path.exists(outputFileName):
				os.remove(outputFileName)

	def setFormat(self, format):
		"""
		Set the format of the picture
		@param format: the format
		@type format: string
		"""
		if format not in ("png", "pdf", "jpeg", "bmp", "tiff"):
			raise Exception("Format '%s' is not supported by RPlotter" % (format))
		self.format = format


	def setWidth(self, width):
		"""
		Set the dimensions of the image produced
		@param width: width of the image
		@type width: int
		"""
		self.width = width
		
		
	def setHeight(self, height):
		"""
		Set the dimensions of the image produced
		@param height: heigth of the image
		@type height: int
		"""
		self.height = height
		
		
	def setImageSize(self, width, height):
		"""
		Set the dimensions of the image produced
		@param width: width of the image
		@type width: int
		@param height: heigth of the image
		@type height: int
		"""
		self.width = width
		self.height = height
		
	def addPlot(self, plot):
		"""
		Add a plot
		@param plots: plot to be included
		@type  plots: L{RPlotter{RPlotter}}
		"""
		self.plots.append(plot)

	def plot(self):
		"""
		Plot the figures
		"""
		scriptHandle = open(self.scriptFileName, "w")
		scriptHandle.write("library(RColorBrewer)\n")
		scriptHandle.write("colorPanel = brewer.pal(n=%d, name=\"Set1\")\n" % (NBCOLORS))
		scriptHandle.write("%s(%s = \"%s\", width = %d, height = %d, bg = \"white\")\n" % (self.format, "filename" if self.format != "pdf" else "file", self.fileName, self.width, self.height))
		scriptHandle.write("par(mfrow=c(%d, 1))\n" % (len(self.plots)))
		for plot in self.plots:
			scriptHandle.write(plot.getScript())
		scriptHandle.write("dev.off()\n")
		scriptHandle.close()
		rCommand = "R"
		if "SMARTRPATH" in os.environ:
			rCommand = os.environ["SMARTRPATH"]
		command = "\"%s\" CMD BATCH %s" % (rCommand, self.scriptFileName)
		status = subprocess.call(command, shell=True)
		if status != 0:
			self.keep = True
			raise Exception("Problem with the execution of script file %s, status is: %s" % (self.scriptFileName, status))

