from structure.transcript import *
from structure.transcriptList import *
from writer.transcriptListWriter import *


class WigWriter(TranscriptListWriter):
  """
  A class that writes a transcript list into a file with WIGGLE format
  @ivar fileName: name of the file 
  @type fileName: string
  @ivar handle: handle to the file
  @type handle: file handle
  @ivar header: first lines of the file
  @type header: string
  """


  def __init__(self, fileName, verbosity = 0):
    """
    Constructor
    @param fileName: name of the file 
    @type fileName: string
    @param verbosity: verbosity
    @type verbosity: int
    """
    self.fileName  = fileName
    self.verbosity = verbosity
    self.data      = {-1: {}, 1: {}}
    self.title     = "Reads"


  def __del__(self):
    """
    Destructor
    Actually print the file
    """
    strand2string = {-1: "-", 1: "+"}
    for strand in self.data:
      fileName = self.fileName
      if fileName.endswith(".wig"):
        fileName = "%s%s.wig" % (fileName[: -len(".wig")], strand2string[strand])
      self.handle = open(fileName, "w")
      self.handle.write("track type=wiggle_0 name=\"%s\"\n" % (self.title))
      for chromosome in sorted(self.data[strand]):
         self.handle.write("variableStep chrom=%s\n" % (chromosome))
         for pos in sorted(self.data[strand][chromosome]):
           self.handle.write("%d\t%d\n" % (pos, self.data[strand][chromosome][pos]))
      self.handle.close()

    
  def setTitle(self, title):
    """
    Set the title of the track
    @param title: the title of the track
    @type  title: string
    """
    if title != None:
      self.title = title


  def copyProperties(self, parser):
    """
    Copy the properties collected by a parser, to produce a similar output
    @param bedParser: a parser
    @type  bedParser: class L{TranscriptListWriter<TranscriptListWriter>}
    """
    self.setTitle(parser.title)
    

  def addTranscript(self, transcript):
    """
    Export the given transcript with GBrowse format
    @param transcript: transcript to be printed
    @type transcript: class L{Transcript<Transcript>}
    @return: a string
    """
    chromosome = transcript.chromosome
    direction  = transcript.direction
    if chromosome not in self.data[direction]:
      self.data[direction][chromosome] = {}
    for exon in transcript.getExons():
      for pos in range(exon.start, exon.end+1):
        if pos not in self.data[direction][chromosome]:
          self.data[direction][chromosome][pos]  = 1
        else:
          self.data[direction][chromosome][pos] += 1
