#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
"""
Read a transcript file and convert it to another format
"""

import os, re
from optparse import OptionParser
from SMART.Java.Python.structure.TranscriptContainer import TranscriptContainer
from commons.core.writer.TranscriptWriter import TranscriptWriter
from SMART.Java.Python.misc.Progress import Progress


class ConvertTranscriptFile(object):
    def __init__(self,inputFileName="", inputFormat ="", outputFileName="", outputFormat="", name="", sequenceFileName=None, strands=False, galaxy=False, feature=None, featurePart=None, verbosity=1):
        self.inputFileName = inputFileName
        self.inputFormat = inputFormat
        self.outputFileName = outputFileName
        self.outputFormat = outputFormat
        self.name = name
        self.sequenceFileName = sequenceFileName
        self.strands = strands
        self.galaxy = galaxy

        self.feature=feature
        self.featurePart=featurePart
        
        self.verbosity = verbosity
         
    def setAttributesFromCmdLine(self):
        description = "Convert Transcript File v1.0.3: Convert a file from a format to another. [Category: Conversion]"
        parser = OptionParser(description = description)
        parser.add_option("-i", "--input",        dest="inputFileName",    action="store",                       type="string", help="input file [compulsory] [format: file in format given by -f]")
        parser.add_option("-f", "--inputFormat",  dest="inputFormat",      action="store",                       type="string", help="format of the input file [compulsory] [format: transcript or mapping file format]")
        parser.add_option("-o", "--output",       dest="outputFileName",   action="store",                       type="string", help="output file [compulsory] [format: output file in format given by -g]")
        parser.add_option("-g", "--outputFormat", dest="outputFormat",     action="store",                       type="string", help="format of the output file [compulsory] [format: transcript file format]")
        parser.add_option("-n", "--name",         dest="name",             action="store",      default="SMART", type="string", help="name for the transcripts [format: string] [default: SMART]")
        parser.add_option("-s", "--sequences",    dest="sequenceFileName", action="store",      default=None,    type="string", help="give the corresponding Multi-Fasta file (useful for EMBL format) [format: string]")
        parser.add_option("-t", "--strands",      dest="strands",          action="store_true", default=False,                  help="consider the 2 strands as different (only useful for writing WIG files) [format: bool] [default: False]")
        parser.add_option("-v", "--verbosity",    dest="verbosity",        action="store",      default=1,       type="int",    help="trace level [format: int] [default: 1]")
        parser.add_option("-G", "--galaxy",       dest="galaxy",           action="store_true", default=False,                  help="used for galaxy [format: bool] [default: False]")
        (options, args) = parser.parse_args()
        self._setAttributesFromOptions(options)

    def _setAttributesFromOptions(self, options):
        self.inputFileName = options.inputFileName
        self.inputFormat = options.inputFormat
        self.outputFileName = options.outputFileName
        self.outputFormat = options.outputFormat
        self.name = options.name  
        self.sequenceFileName = options.sequenceFileName
        self.strands = options.strands
        self.galaxy =  options.galaxy
        self.verbosity = options.verbosity

    def run(self):
        # create parser
        parser = TranscriptContainer(self.inputFileName, self.inputFormat, self.verbosity)
        # create writer
        writer = TranscriptWriter(self.outputFileName, self.outputFormat, self.verbosity)
        # connect parser and writer
        writer.setContainer(parser)
            
        if self.name != None:
            writer.setTitle(self.name)
        if self.feature != None:
            writer.setFeature(self.feature)
        if self.featurePart != None:
            writer.setFeaturePart(self.featurePart)
        if self.sequenceFileName != None:
            writer.addSequenceFile(self.sequenceFileName)
            
        nbItems = 0
        if self.verbosity > 0:
            nbItems = parser.getNbItems()
            print "%i items found" % (nbItems)
    
        if self.strands:
            writer.setStrands(True)
        # convert
        writer.write()
        writer.close()

if __name__ == "__main__":
    iConvertTranscriptFile = ConvertTranscriptFile()
    iConvertTranscriptFile.setAttributesFromCmdLine()
    iConvertTranscriptFile.run()
