#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
"""
Plot the data from the data files
"""
import sys
import math
from optparse import OptionParser
from SMART.Java.Python.structure.TranscriptContainer import TranscriptContainer
from SMART.Java.Python.misc.RPlotter import RPlotter


class PlotTranscriptList(object):

    def __init__(self, verbosity = 0):
        self.inputFileName    = None
        self.format                 = None
        self.x                            = None
        self.y                            = None
        self.z                            = None
        self.xDefault             = None
        self.yDefault             = None
        self.zDefault             = None
        self.xLabel                 = None
        self.yLabel                 = None
        self.shape                    = None
        self.bucket                 = None
        self.keep                     = None
        self.log                        = None
        self.verbosity            = None


    def setPlotter(self, outputFileName, keep, log, xLabel, yLabel):
        self.plotter = RPlotter(outputFileName, self.verbosity, keep)
        if self.shape != "barplot":
            self.plotter.setLog(log)
        self.plotter.setXLabel(xLabel)
        self.plotter.setYLabel(yLabel)


    def setShape(self, shape):
        if self.shape == "line":
            pass
        elif shape == "barplot":
            self.plotter.setBarplot(True)
        elif shape == "points":
            self.plotter.setPoints(True)
        elif shape == "heatPoints":
            self.plotter.setHeatPoints(True)
        else:
            sys.exit("Do not understand shape '%s'" % (shape))


    def setInput(self, inputFileName, format):
        self.parser = TranscriptContainer(inputFileName, format, self.verbosity)


    def getValues(self, transcript):
        x, y, z = None, None, None
        x = transcript.getTagValue(self.x)
        if self.y != None:
            y = transcript.getTagValue(self.y)
        if self.z != None:
            z = transcript.getTagValue(self.z)
        if x == None:
            if self.xDefault != None:
                x = self.xDefault
            else:
                sys.exit("Error! Transcript %s do not have the x-tag %s" % (transcript, self.x))
        if y == None and self.shape != "line" and self.shape != "barplot":
            if self.yDefault != None:
                y = self.yDefault
            else:
                sys.exit("Error! Transcript %s do not have the y-tag %s" % (transcript, self.y))
        if self.z != None:
            if z == None:
                if self.zDefault != None:
                    z = self.zDefault
                else:
                    sys.exit("Error! Transcript %s do not have the z-tag %s" % (transcript, self.z))
        x = float(x)
        if self.y != None:
            y = float(y)
        if self.z != None:
            z = float(z)
        return (x, y, z)


    def readFile(self):
        cpt            = 1
        line         = {}
        heatLine = {}
        for transcript in self.parser.getIterator():
            x, y, z = self.getValues(transcript)

            name = transcript.name
            if name == "unnamed transcript":
                name = "transcript %d" % (cpt)
                cpt += 1
            if self.shape == "points":
                line[name] = (x, y)
            elif self.shape == "heatPoints":
                line[name] = (x, y)
                heatLine[name] = z
            elif self.shape == "line" or self.shape == "barplot":
                if x not in line:
                    line[x] = 1
                else:
                    line[x] += 1
            else:
                sys.exit("Do not understand shape '%s'" % (self.shape))
        return line, heatLine


    def putLineInBuckets(self, line):
        tmpLine = line
        line        = {}
        for key, value in tmpLine.iteritems():
            line[int(key / float(self.bucket)) * self.bucket] = value
        return line


    def clusterInBarplot(self, line):
        nbZeros        = 0
        minValue     = min(line.keys())
        maxValue     = max(line.keys())
        if self.log != "":
            if minValue == 0:
                minValue = 1000000000
                for value in line:
                    if value < minValue:
                        if value == 0:
                            nbZeros += 1
                        else:
                            minValue = value
            minValue = math.log(minValue)
            maxValue = math.log(maxValue)
        bucketSize = (maxValue - minValue) / self.bucket
        tmpLine        = line
        line             = {}
        for i in range(int(self.bucket) + 1):
            line[i * bucketSize + minValue] = 0
        for key, value in tmpLine.iteritems():
            if self.log != "" and key != 0:
                key = math.log(key)
            bucketKey = int((key - minValue) / bucketSize) * bucketSize + minValue
            if self.log == "" or key != 0:
                line[bucketKey] += value
#     if self.log != "":
#         tmpLine = line
#         line        = {}
#         for key, value in tmpLine.iteritems():
#             line[math.exp(key)] = value
        print "%d zeros have been removed" % (nbZeros)
        return line


    def getSpearmanRho(self):
        rho = self.plotter.getSpearmanRho()
        if rho == None:
            print "Cannot compute Spearman rho."
        else:
            print "Spearman rho: %f" % (rho)


    def run(self):
        line, heatLine = self.readFile()

        if self.shape == "line" and self.bucket != None:
            line = self.putLineInBuckets(line)
        if self.shape == "barplot":
            line = self.clusterInBarplot(line)

        if self.shape == "points" or self.shape == "barplot" or self.shape == "line":
            self.plotter.addLine(line)
        elif self.shape == "heatPoints":
            self.plotter.addLine(line)
            self.plotter.addHeatLine(heatLine)
        else:
            sys.exit("Do not understand shape '%s'" % (self.shape))

        self.plotter.plot()

        if self.shape == "points" or self.shape == "heatPoints":
            self.getSpearmanRho()



if __name__ == "__main__":
    
    # parse command line
    description = "Plot v1.0.2: Plot some information from a list of transcripts. [Category: Visualization]"

    parser = OptionParser(description = description)
    parser.add_option("-i", "--input",dest="inputFileName", action="store", type="string", help="input file [compulsory] [format: file in transcript format given by -f]")
    parser.add_option("-f", "--format",dest="format", action="store",type="string", help="format of the input [compulsory] [format: transcript file format]")
    parser.add_option("-x", "--x",dest="x",action="store", type="string", help="tag for the x value [format: string]")
    parser.add_option("-y", "--y",dest="y",action="store", type="string", help="tag for the y value [format: string]")
    parser.add_option("-z", "--z",dest="z", action="store", default=None,type="string", help="tag for the z value [format: string]")
    parser.add_option("-X", "--xDefault",dest="xDefault",action="store", default=None,type="float",help="value for x when tag is not present [format: float]")
    parser.add_option("-Y", "--yDefault",dest="yDefault",action="store",default=None,type="float",help="value for y when tag is not present [format: float]")
    parser.add_option("-Z", "--zDefault",dest="zDefault", action="store",default=None,type="float",help="value for z when tag is not present [format: float]")
    parser.add_option("-n", "--xLabel",dest="xLabel",action="store",default="",type="string", help="label on the x-axis [format: string] [default: ]")
    parser.add_option("-m", "--yLabel",dest="yLabel",action="store",default="", type="string", help="label on the y-axis [format: string] [default: ]")
    parser.add_option("-o", "--output",dest="outputFileName",action="store",type="string", help="output file names [format: output file in PNG format]")
    parser.add_option("-s", "--shape",dest="shape",action="store", type="string", help="shape of the plot [format: choice (barplot, line, points, heatPoints)]")
    parser.add_option("-b", "--bucket",dest="bucket",action="store",default=None,type="float",help="bucket size (for the line plot) [format: int] [default: 1]")
    parser.add_option("-k", "--keep",dest="keep",action="store_true", default=False, help="keep temporary files [format: bool]")
    parser.add_option("-l", "--log",dest="log",action="store",default="",type="string", help="use log on x- or y-axis (write 'x', 'y' or 'xy') [format: string] [default: ]")
    parser.add_option("-v", "--verbosity",dest="verbosity",action="store",default=1, type="int",help="trace level [format: int]")
    (options, args) = parser.parse_args()

    plotTranscriptList = PlotTranscriptList(options.verbosity)
    plotTranscriptList.x                = options.x
    plotTranscriptList.y                = options.y
    plotTranscriptList.z                = options.z
    plotTranscriptList.xDefault = options.xDefault
    plotTranscriptList.yDefault = options.yDefault
    plotTranscriptList.zDefault = options.zDefault
    plotTranscriptList.shape        = options.shape
    plotTranscriptList.bucket     = options.bucket
    plotTranscriptList.log            = options.log
    plotTranscriptList.setPlotter(options.outputFileName, options.keep, options.log, options.xLabel, options.yLabel)
    plotTranscriptList.setShape(options.shape)
    plotTranscriptList.setInput(options.inputFileName, options.format)
    plotTranscriptList.run()

