import re
import sys
from structure.transcript import *
from structure.transcriptList import *


class TranscriptListWriter(object):
  """
  An interface that writes a transcript list into a file
  @ivar fileName: name of the file 
  @type fileName: string
  @ivar handle: handle to the file
  @type handle: file handle
  @ivar header: first lines of the file
  @type header: string
  @ivar started: whether some transcripts have already been writted
  @type started: boolean
  @ivar lastChromosome: the chromosome on which the transcript which was inserted last
  @type lastChromosome: string
  """


  def __init__(self, fileName, verbosity = 0):
    """
    Constructor
    @param fileName: name of the file 
    @type fileName: string
    @param verbosity: verbosity
    @type verbosity: int
    """
    self.fileName       = fileName
    self.verbosity      = verbosity
    self.handle         = open(self.fileName, "w")
    self.started        = False
    self.lastChromosome = None
    self.header         = ""


  def __del__(self):
    """
    Destructor
    """
    self.handle.close()


  def addTranscript(self, transcript):
    """
    Add a transcript to the list of transcripts to be written
    @param transcript: transcript to be written
    @type  transcript: class L{Transcript<Transcript>}
    """
    if not self.started:
      self.handle.write(self.header)
      self.started = True

    self.handle.write(self.printTranscript(transcript))
    self.lastChromosome = transcript.chromosome


  def addElement(self, element):
    """
    Same as "addTranscript"
    @param element: transcript to be written
    @type  element: class L{Transcript<Transcript>}
    """
    self.addTranscript(element)


  def addTranscriptList(self, transcriptList):
    """
    Add a list of transcripts to the transcripts to be written
    @param transcriptList: transcripts to be written
    @type  transcriptList: class L{TranscriptList<TranscriptList>}
    """
    for transcript in transcriptList.getIterator():
      self.addTranscript(transcript)
      

  def addTranscriptTable(self, transcriptTable):
    """
    Add a list of transcripts in a mySQL table to the transcripts to be written
    @param transcriptTable: transcripts to be written
    @type  transcriptTable: class L{MySqlTranscriptTable<MySqlTranscriptTable>}
    """
    for transcript in transcriptTable.getIterator():
      self.addTranscript(transcript)
      
      
  def setTitle(self, title):
    """
    Possibly write a title for the list (by default, do nothing)
    @param title: a title for the list
    @type title:  string
    """
    pass


  def write(self):
    """
    No-op
    """
    pass
