import sys
import math
from structure import bins


class Interval(object):
  """
  Store a genomic interval
  @ivar name:          name of the interval [optional]
  @type name:          string
  @ivar chromosome:    chromosome on which the interval is
  @type chromosome:    string
  @ivar start:         start point of the interval
  @type start:         int
  @ivar end:           end point
  @type end:           int [computed if start and size are given]
  @ivar size:          size of the interval
  @type size:          int [computed if start and end are given]
  @ivar direction:     direction of the interval (+ / -)
  @type direction:     int (1 / -1)
  @ivar id:            id of the interval [optional]
  @type id:            int
  @ivar bin:           bin in which the interval should be if stored in a database [computed]
  @type bin:           int 
  @ival tags:          information about the transcript [optional]
  @type tags:          dict
  @ivar verbosity:     verbosity
  @type verbosity:     int [default: 0]
  """

  def __init__(self, interval = None, verbosity = 0):
    """
    Constructor
    @param interval:  interval to be copied
    @type  interval:  class L{Interval<Interval>}
    @param verbosity: verbosity
    @type  verbosity: int
    """
    self.name          = None
    self.chromosome    = None
    self.start         = None
    self.end           = None
    self.size          = None
    self.direction     = 0
    self.id            = None
    self.bin           = None
    self.verbosity     = verbosity
    self.tags          = {}
    if interval != None:
      self.copy(interval)


  def copy(self, interval):
    """
    Copy method
    @param interval: interval to be copied
    @type  interval: class L{Interval<Interval>}
    """
    self.name          = interval.name     
    self.chromosome    = interval.chromosome     
    self.start         = interval.start    
    self.end           = interval.end      
    self.size          = interval.size     
    self.direction     = interval.direction
    self.id            = interval.id
    self.bin           = interval.bin
    self.tags          = {}
    for tag in interval.tags:
      self.tags[tag] = interval.tags[tag]
    self.verbosity     = interval.verbosity


  def setName(self, name):
    """
    Set the name
    @param name: name of the interval
    @type  name: string
    """
    self.name = name


  def setChromosome(self, chromosome):
    """
    Set the chromosome
    @param chromosome: chromosome on which the interval is
    @type  chromosome: string
    """
    self.chromosome = chromosome.replace(".", "_").replace("|", "_")


  def setStart(self, start):
    """
    Set the start point
    Possibly modify size and reset bin
    @param start: start point of the interval
    @type  start: int
    """
    self.start = start
    if self.end != None and self.size == None:
      self.size = self.end - self.start + 1
    self.bin = None


  def setEnd(self, end):
    """
    Set the end point
    Possibly modify size and reset bin
    @param end: end point of the interval of the interval
    @type  end: int
    """
    self.end = end
    if self.start != None and self.size == None:
      self.size = self.end - self.start + 1
    self.bin = None


  def setSize(self, size):
    """
    Set the size
    Possibly modify the end point
    @param size: size of the transcript
    @type  size: int
    """
    self.size = size
    if self.end == None and self.start != None:
      self.end = self.start + self.size - 1


  def getSize(self):
    """
    Get the size
    """
    return self.size


  def setDirection(self, direction):
    """
    Set the direction of the interval
    Possibly parse different formats
    @param direction: direction of the transcript (+ / -)
    @type  direction: int or string
    """
    if type(direction).__name__ == 'int':
      self.direction = direction / abs(direction)
    elif type(direction).__name__ == 'str':
      if direction == "+":
        self.direction = 1
      elif direction == "-":
        self.direction = -1
      elif direction == "1" or direction == "-1":
        self.direction = int(direction)
      elif direction.lower() == "plus":
        self.direction = 1
      elif direction.lower() == "minus":
        self.direction = -1
      else:
        sys.exit("Cannot understand direction " + direction)
    else:
      sys.exit("Cannot understand direction " + direction)
      
  
  def setTagValue(self, name, value):
    """
    Set a tag
    @param name:  name of the tag
    @type  name:  string
    @param value: value of the tag
    @type  value: int or string
    """
    self.tags[name] = value
    
    
  def getTagNames(self):
    """
    Get all the names of the tags
    """
    return self.tags.keys()


  def getTagValue(self, tag):
    """
    Get the value of a tag
    @param tag: name of a tag
    @type  tag: string
    """
    if tag not in self.tags:
      return None
    return self.tags[tag]


  def getTagValues(self, tagSep = "; ", fieldSep = " "):
    """
    Get the formatted tag values
    @param tagSep:   separator between tags
    @type  tagSep:   string
    @param fieldSep: separator between tag name and tag value
    @type  fieldSep: string
    """
    tags = []
    for name in self.tags:
      value = self.tags[name]
      if isinstance(value, basestring):
        tags.append("%s%s%s" % (name, fieldSep, self.tags[name]))
      else:
        tags.append("%s%s%i" % (name, fieldSep, self.tags[name]))
    return tagSep.join(tags)

  
  def setTagValues(self, tags, tagSep = "; ", fieldSep = " "):
    """
    Set the tag values using given string
    @param tags:     the tags, concatenated
    @type  tags:     string
    @param tagSep:   separator between tags
    @type  tagSep:   string
    @param fieldSep: separator between tag name and tag value
    @type  fieldSep: string
    """
    if tags == "":
      self.tags = {}
      return
    for splittedTag in tags.split(tagSep):
      pos = splittedTag.find(fieldSep)
      if pos == -1:
        sys.exit("Weird field '%s' in tags '%s'" % (splittedTag, tags))
      tag   = splittedTag[:pos]
      value = splittedTag[pos+1:]
      try:
      	intValue       = int(value)
        self.tags[tag] = intValue
      except ValueError:
        self.tags[tag] = value


  def deleteTag(self, tag):
    """
    Remove a tag
    @param tag: the tag to be removed
    @type  tag: string
    """
    if tag in self.tags:
      del self.tags[tag]

  
  def setNbOccurrences(self, nbOccurrences):
    """
    Set the number of occurrences of the interval
    @param nbOccurrences: number of occurrences of the interval
    @type  nbOccurrences: int
    """
    self.setTagValue("nbOccurrences", nbOccurrences)
    
      
  def setOccurrence(self, occurrence):
    """
    Set the occurrence of this interval
    @param occurrence: an occurrence for this transcript
    @type  occurrence: int
    """
    self.setTagValue("occurrence", occurrence)
    
#  def __eq__(self, interval):
#    """
#    Whether two intervals are equal (start and end at same position)
#    @param interval: object to be compared to
#    @type  interval: class L{Interval<Interval>}
#    """
#    return self.start == interval.start and self.end == interval.end
#  
#  
#  def __lt__(self, interval):
#    """
#    Whether one interval is before the other (start is before or starts are the same and end is before)
#    @param interval: object to be compared to
#    @type  interval: class L{Interval<Interval>}
#    """
#    return self.start < interval.start or (self.start == interval.start and self.end < interval.end)
#
#
#  def __le__(self, interval):
#    """
#    Whether one interval is not after the other (before or equal)
#    @param interval: object to be compared to
#    @type  interval: class L{Interval<Interval>}
#    """
#    return self < interval or self == interval
#
#
#  def __gt__(self, interval):
#    """
#    Whether one interval is after the other (start is after or starts are the same and end is after)
#    @param interval: object to be compared to
#    @type  interval: class L{Interval<Interval>}
#    """
#    return self.start > interval.start or (self.start == interval.start and self.end > interval.end)
#
#
#  def __ge__(self, interval):
#    """
#    Whether one interval is not before the other (after or equal)
#    @param interval: object to be compared to
#    @type  interval: class L{Interval<Interval>}
#    """
#    return self > interval or self == interval


  def overlapWith(self, interval):
    """
    Whether two intervals overlap
    @param interval: object to be compared to
    @type  interval: class L{Interval<Interval>}
    """    
    if self.chromosome != interval.chromosome:
      return False
    return (self.start <= interval.end) and (self.end >= interval.start)
  
  
  def getDistance(self, interval):
    """
    Get the distance between two intervals (a non-negative value)
    @param interval: another interval
    @type  interval: class L{Interval<Interval>}
    """    
    if self.overlapWith(interval):
      return 0
    if self.chromosome != interval.chromosome:
      sys.exit("Cannot get the distance between %s and %s" % (str(self), str(interval)))
    return min(abs(self.start - interval.end), abs(self.end - interval.start))


  def getRelativeDistance(self, interval):
    """
    Get the distance between two intervals (negative if first interval is before)
    @param interval: another interval
    @type  interval: class L{Interval<Interval>}
    """    
    if self.overlapWith(interval):
      return 0
    if self.chromosome != interval.chromosome:
      sys.exit("Cannot get the distance between %s and %s" % (str(self), str(interval)))
    if self.end < interval.start:
      distance = interval.start - self.end
      #distance = self.end - interval.start
    else:
      distance = interval.end - self.start
      #distance = self.start - interval.end
    distance *= self.direction
    return distance


  def merge(self, interval):
    """
    Merge two intervals
    @param interval: another interval
    @type  interval: class L{Interval<Interval>}
    """    
    if self.chromosome != interval.chromosome or self.direction != interval.direction:
      sys.exit("Cannot merge %s and %s" % (str(self), str(interval)))
    self.setStart(min(self.start, interval.start))
    self.setEnd(max(self.end, interval.end))
    self.size = self.end - self.start + 1
    if "nbElements" not in self.getTagNames():
      self.setTagValue("nbElements", 1)
    if "nbElements" not in interval.getTagNames():
      interval.setTagValue("nbElements", 1)
    self.setTagValue("nbElements", self.getTagValue("nbElements") + interval.getTagValue("nbElements"))
    self.bin = None
    for tagName in ("identity", "nbOccurrences", "occurrence", "nbMismatches", "nbGaps", "rank", "evalue", "bestRegion"):
      if tagName in self.getTagNames():
        del self.tags[tagName]


  def getBin(self):
    """
    Get the bin of the interval
    Computed on the fly
    """
    if self.bin == None:
      self.bin = bins.getBin(self.start, self.end)
    return self.bin


  def getBins(self):
    """
    Get all the bin this interval could fall into
    """
    return bins.getOverlappingBins(self.start, self.end)


  def getSqlVariables(cls):
    """
    Get the properties of the object that should be saved in a database
    """
    variables = ["name", "chromosome", "start", "end", "direction", "size", "tags", "bin"]
    return variables
  getSqlVariables = classmethod(getSqlVariables)


  def setSqlValues(self, array):
    """
    Set the values of the properties of this object as given by a results line of a SQL query
    """
    self.id         = array[0]
    self.name       = array[1]
    self.chromosome = array[2]
    self.start      = array[3]
    self.end        = array[4]
    self.direction  = array[5]
    self.size       = array[6]
    self.setTagValues(array[7], ";", "=")
    self.bin        = array[8]


  def getSqlValues(self):
    """
    Get the values of the properties that should be saved in a database
    """
    values = dict()
    values["name"]       = self.name
    values["chromosome"] = self.chromosome
    values["start"]      = self.start
    values["end"]        = self.end
    values["direction"]  = self.direction
    values["size"]       = self.size
    values["tags"]       = self.getTagValues(";", "=")
    values["bin"]        = self.getBin()
    return values


  def getSqlTypes(cls):
    """
    Get the values of the properties that should be saved in a database
    """
    types = dict()
    types["name"]       = "varchar"
    types["chromosome"] = "varchar"
    types["start"]      = "int"
    types["end"]        = "int"
    types["direction"]  = "tinyint"
    types["size"]       = "int"
    types["tags"]       = "varchar"
    types["bin"]        = "int"
    return types
  getSqlTypes = classmethod(getSqlTypes)
  

  def getSqlSizes(cls):
    """
    Get the sizes of the properties that should be saved in a database
    """
    sizes = dict()
    sizes["name"]          = 255
    sizes["chromosome"]    = 255
    sizes["start"]         = 11
    sizes["end"]           = 11
    sizes["direction"]     = 4
    sizes["size"]          = 11
    sizes["tags"]          = 1023
    sizes["bin"]           = 11
    return sizes
  getSqlSizes = classmethod(getSqlSizes)
  

  def printCoordinates(self):
    """
    Print start and end positions (depending on the direction of the interval)
    """
    if self.direction == 1:
      return "%d-%d" % (self.start, self.end)
    else:
      return "%d-%d" % (self.end, self.start)

  
  def extractSequence(self, parser):
    """
    Get the sequence corresponding to this interval
    @param parser: a parser to a FASTA file
    @type  parser: class L{SequenceListParser<SequenceListParser>}
    @return      : a instance of L{Sequence<Sequence>}
    """
    return parser.getSubSequence(self.chromosome, self.start, self.end, self.direction, self.name)
  
  
  def extractWigData(self, parser):
    """
    Get the data retrieved from a wig file
    @param parser: a parser class to a WIG file
    @type  parser: class L{WigParser<WigParser>}
    """
    return parser.getRange(self.chromosome, self.start, self.end)


  def __str__(self):
    """
    Output a simple representation of this interval
    """
    direction = "+"
    if self.direction == -1:
      direction = "-"
    string = "%s:%d-%d (%s)" % (self.chromosome, self.start, self.end, direction)
    if self.name != "":
      string = "(%s) %s" % (self.name, string)
    return string

