#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import sys
from commons.core.parsing.MapperParser import MapperParser
from SMART.Java.Python.structure import Mapping

class ElandParser(MapperParser):
    """A class that parses ELAND format"""

    def __init__(self, fileName, verbosity = 0):
        super(ElandParser, self).__init__(fileName, verbosity)


    def __del__(self):
        super(ElandParser, self).__del__()


    def getFileFormats():
        return ["eland"]
    getFileFormats = staticmethod(getFileFormats)


    def skipFirstLines(self):
        pass


    def getInfos(self):
        super(ElandParser, self).getInfos()
        

    def parseLine(self, line):

        line = line.strip()

        fields = line.split("\t")
        if len(fields) < 22:
            sys.exit("Line %d '%s' does not look like a ELAND line (number of fields is %d instead of 22)" % (self.currentLineNb, line, len(fields)))

        flowCell = fields[0]
        run = fields[1]
        lane = fields[2]
        tile = fields[3]
        xcoord = fields[4]
        ycoord = fields[5]
        index = fields[6]
        number = fields[7]
        read = fields[8]
        quality = fields[9]
        chromosome = fields[10]
        contig = fields[11]
        position = fields[12]
        strand = fields[13]
        description = fields[14]
        singleScore = fields[15]
        pairScore = fields[16]
        partnerChromosome = fields[17]
        partnerContig = fields[18]
        partnerOffset = fields[19]
        partnerStrand = fields[20]
        filtering = fields[21]

        if number != "1":
            sys.exit("S-MART cannot handle pair-end reads yet!")

        # nothing found
        if position == "":
            return None

        name = "%s_%s:%s:%s:%s:%s#0/1" % (flowCell, run, lane, tile, xcoord, ycoord)
        direction = 1 if strand == "F" else -1
        nbMismatches = 0
        for char in description:
            if ord("A") <= ord(char) and ord(char) <= ord("Z"):
                nbMismatches += 1

        mapping = Mapping()
        mapping.setTagValue("qualityString", quality)
        
        mapping.queryInterval.setName(name)
        mapping.queryInterval.setDirection(direction)
        mapping.queryInterval.setStart(1)
        mapping.queryInterval.setEnd(len(read))

        mapping.targetInterval.setChromosome(chromosome)
        mapping.targetInterval.setStart(int(position))
        mapping.targetInterval.setEnd(int(position) + len(read))
        mapping.targetInterval.setDirection(1)

        mapping.setSize(len(read))
        mapping.setDirection(direction)

        mapping.setNbGaps(0)
        mapping.setNbMismatches(nbMismatches)
        mapping.setTagValue("score", int(singleScore))

        if filtering == "Y":
            return mapping
        # mapping filtered out
        return None
