# Copyright INRA (Institut National de la Recherche Agronomique)
# http://www.inra.fr
# http://urgi.versailles.inra.fr
#
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software.  You can  use, 
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info". 
#
# As a counterpart to the access to the source code and  rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty  and the software's author,  the holder of the
# economic rights,  and the successive licensors  have only  limited
# liability. 
#
# In this respect, the user's attention is drawn to the risks associated
# with loading,  using,  modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean  that it is complicated to manipulate,  and  that  also
# therefore means  that it is reserved for developers  and  experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or 
# data to be ensured and,  more generally, to use and operate it in the 
# same conditions as regards security. 
#
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.


import unittest
import os
import ConfigParser
from commons.core.checker.CheckerUtils import CheckerUtils
from commons.core.checker.CheckerException import CheckerException
from ConfigParser import NoOptionError
from ConfigParser import NoSectionError

class Test_CheckerUtils( unittest.TestCase ):
    
    def setUp(self):
        self.queueFileName = "queueName.txt"
        self.configFileName = "dummyConfig.cfg"
    
    def tearDown(self):
        if os.path.exists(self.queueFileName):
            os.remove(self.queueFileName)
        if os.path.exists(self.configFileName):
            os.remove(self.configFileName)
    
    def test_isBlastNameInBlastValues( self ):
        correctValueList = [ "blastn", "blastp", "blastx", "tblastn", "tblastx" ]
        for value in correctValueList:
            self.assertFalse( CheckerUtils.isBlastNameNotInBlastValues( value ) )
            
        incorrectValueList = [ "badbalst", "wublast" ]
        for value in incorrectValueList:
            self.assertTrue( CheckerUtils.isBlastNameNotInBlastValues( value ) )
            
    def test_isNotTRUEisNotFALSE( self ):
        correctValueList = [ "TRUE", "FALSE" ]
        for value in correctValueList:
            self.assertFalse( CheckerUtils.isNotTRUEisNotFALSE( value ) )
            
        incorrectValueList = [ "True", "False" ]
        for value in incorrectValueList:
            self.assertTrue( CheckerUtils.isNotTRUEisNotFALSE( value ) )
            
    def test_isRessourceNotExists( self ):
        fileName = "dummyFile.txt"
        self.assertTrue( CheckerUtils.isRessourceNotExits( fileName ) )
        os.system( "touch %s" % ( fileName ) )
        self.assertFalse( CheckerUtils.isRessourceNotExits( fileName ) )
        os.remove( fileName )
        
    def test_isNotAeValueWithOneDigit2DecimalsAtLeast( self ):
        correctEValueList = [ "5e-32", "7e-45", "1e-2122", "9e-32" ]
        for value in correctEValueList:
            self.assertFalse( CheckerUtils.isNotAeValueWithOneDigit2DecimalsAtLeast( value ) )
            
        incorrecEValueStr = [ "10e-32", "2e-3", "2e-2", "1", "cxhhe" ]
        for value in incorrecEValueStr:
            self.assertTrue( CheckerUtils.isNotAeValueWithOneDigit2DecimalsAtLeast( value ) )
            
    def test_isNotADigit( self ):
        correctDigitList = [ "1", "21", "345" ]
        for value in correctDigitList:
            self.assertFalse( CheckerUtils.isNotANumber( value ) )
            
        incorrectDigitList = [ "A", "dfdf", "((((" ]
        for value in incorrectDigitList:
            self.assertTrue( CheckerUtils.isNotANumber( value ) )
            
    def test_isExecutableInUserPath( self ):
        exeName = "ls"
        self.assertTrue( CheckerUtils.isExecutableInUserPath( exeName ) )

        exeName = "hsvegdtefaobfbcheta"
        self.assertFalse( CheckerUtils.isExecutableInUserPath( exeName ) )
    
    def test_isExecutableInUserPath_case_endsWith( self ):
        exeName = "s"
        self.assertFalse( CheckerUtils.isExecutableInUserPath( exeName ) )

    def test_isQueueNameValid_valid(self):
        cmd = "qconf -sql > " + self.queueFileName
        os.system(cmd)
        f = open(self.queueFileName, "r")
        lQueueNames = f.readlines()
        f.close()
        if lQueueNames == []:
            print "Warning you need SGE installed\n"
        else:
            self.assertTrue(CheckerUtils.isQueueNameValid(lQueueNames[0].strip()))
        
    def test_isQueueNameValid_not_valid(self):
        cmd = "qconf -sql > " + self.queueFileName
        os.system(cmd)
        f = open(self.queueFileName, "r")
        lQueueNames = f.readlines()
        f.close()
        if lQueueNames == []:
            print "Warning you need SGE installed\n"
        else:
            self.assertFalse(CheckerUtils.isQueueNameValid("dummyQueueName"))

    def test_isQueueNameValid_valid_with_parameter(self):
        cmd = "qconf -sql > " + self.queueFileName
        os.system(cmd)
        f = open(self.queueFileName, "r")
        lQueueNames = f.readlines()
        f.close()
        if lQueueNames == []:
            print "Warning you need SGE installed\n"
        else:
            queueNameWithParameter = lQueueNames[0].strip() + " test=TRUE"
            self.assertTrue(CheckerUtils.isQueueNameValid(queueNameWithParameter))
            
    def test_isCharAlphanumOrUnderscore_valid(self):
        project_name = "Project_Name"
        self.assertTrue(CheckerUtils.isCharAlphanumOrUnderscore(project_name))
        
    def test_isCharAlphanumOrUnderscore_not_valid(self):
        project_name = "Project_Name,"
        self.assertFalse(CheckerUtils.isCharAlphanumOrUnderscore(project_name))
        
    def test_isMax15Char_valid(self):
        project_name = "Project_Name"
        self.assertTrue(CheckerUtils.isMax15Char(project_name))
        
    def test_isMax15Char_not_valid(self):
        project_name = "Project_Name_tooLong"
        self.assertFalse(CheckerUtils.isMax15Char(project_name))
        
    def test_checkSectionInConfigFile_empty_section(self):
        sectionName = "sectionName"
        line = "[" + sectionName + "]"
        configFile = open(self.configFileName, "w")
        configFile.write(line)
        configFile.close()
        configFile = open(self.configFileName, "r")
        
        config = ConfigParser.ConfigParser()
        config.readfp(configFile)
        
        isSectionExceptionRaised = False
        try:
            CheckerUtils.checkSectionInConfigFile(config, sectionName)
        except NoSectionError:
            isSectionExceptionRaised = True
        configFile.close()
        self.assertFalse(isSectionExceptionRaised)
    
    def test_checkSectionInConfigFile(self):
        sectionName = "sectionName"
        lines = "[" + sectionName + "]\ndummyOption : dummyOption\n"
        configFile = open(self.configFileName, "w")
        configFile.write(lines)
        configFile.close()
        configFile = open(self.configFileName, "r")
        
        config = ConfigParser.ConfigParser()
        config.readfp(configFile)
        
        isSectionExceptionRaised = False
        try:
            CheckerUtils.checkSectionInConfigFile(config, sectionName)
        except NoSectionError:
            isSectionExceptionRaised = True
        configFile.close()
        self.assertFalse(isSectionExceptionRaised)
        
    def test_checkSectionInConfigFile_not_valid(self):
        sectionName = "sectionName"
        line = " "
        configFile = open(self.configFileName, "w")
        configFile.write(line)
        configFile.close()
        configFile = open(self.configFileName, "r")
        
        config = ConfigParser.ConfigParser()
        config.readfp(configFile)
        
        isSectionExceptionRaised = False
        try:
            CheckerUtils.checkSectionInConfigFile(config, sectionName)
        except NoSectionError:
            isSectionExceptionRaised = True
        configFile.close()
        self.assertTrue(isSectionExceptionRaised)
        
    def test_checkOptionInSectionInConfigFile(self):
        optionName = "optionName"
        sectionName = "sectionName"
        configFile = open(self.configFileName, "w")
        configFile.write("[" + sectionName + "]\n")
        configFile.write(optionName + ": ")
        configFile.close()
        configFile = open(self.configFileName, "r")
        
        config = ConfigParser.ConfigParser()
        config.readfp(configFile)
        
        isSectionExceptionRaised = False
        try:
            CheckerUtils.checkOptionInSectionInConfigFile(config, sectionName, optionName)
        except NoOptionError:
            isSectionExceptionRaised = True
        configFile.close()
        self.assertFalse(isSectionExceptionRaised)
        
    def test_checkOptionInSectionInConfigFile_not_present(self):
        optionName = " "
        sectionName = "sectionName"
        configFile = open(self.configFileName, "w")
        configFile.write("[" + sectionName + "]\n")
        configFile.write(optionName)
        configFile.close()
        configFile = open(self.configFileName, "r")
        
        config = ConfigParser.ConfigParser()
        config.readfp(configFile)
        
        isSectionExceptionRaised = False
        try:
            CheckerUtils.checkOptionInSectionInConfigFile(config, sectionName, optionName)
        except NoOptionError:
            isSectionExceptionRaised = True
        configFile.close()
        self.assertTrue(isSectionExceptionRaised)
        
    def test_checkOptionInSectionInConfigFile_not_in_the_right_section(self):
        optionName = "optionName"
        rightSection = "dummySection2"
        configFile = open(self.configFileName, "w")
        configFile.write("[dummySection1]\n")
        configFile.write(optionName + ": \n")
        configFile.write("[" + rightSection + "]")
        configFile.close()
        configFile = open(self.configFileName, "r")
        
        config = ConfigParser.ConfigParser()
        config.readfp(configFile)
        
        isSectionExceptionRaised = False
        try:
            CheckerUtils.checkOptionInSectionInConfigFile(config, rightSection, optionName)
        except NoOptionError:
            isSectionExceptionRaised = True
        configFile.close()
        self.assertTrue(isSectionExceptionRaised)
        
    def test_checkConfigVersion_different_version_between_changeLog_and_configFile(self):
        changeLogFileName = "dummyCHANGELOG"
        
        self._writeChangeLogFile(changeLogFileName)
        self._writeConfigFile("repet_version: 1.3.5\n")
        
        changeLogFileHandler = open(changeLogFileName, "r")
        configFileHandler = open(self.configFileName, "r")
        config = ConfigParser.ConfigParser()
        config.readfp(configFileHandler)
        
        self.assertRaises(CheckerException, CheckerUtils.checkConfigVersion, changeLogFileHandler, config)
        
        changeLogFileHandler.close()
        configFileHandler.close()
        os.remove(changeLogFileName)
        
    def test_checkConfigVersion_same_version_between_changeLog_and_configFile(self):
        changeLogFileName = "dummyCHANGELOG"
        self._writeChangeLogFile(changeLogFileName)
        self._writeConfigFile("repet_version: 1.3.6\n")
        
        changeLogFileHandler = open(changeLogFileName, "r")
        configFileHandler = open(self.configFileName, "r")
        config = ConfigParser.ConfigParser()
        config.readfp(configFileHandler)
        
        isCheckerExceptionRaised = False
        try:
            CheckerUtils.checkConfigVersion(changeLogFileHandler, config)
        except CheckerException:
            isCheckerExceptionRaised = True
        self.assertFalse(isCheckerExceptionRaised)
        
        changeLogFileHandler.close()
        configFileHandler.close()
        os.remove(changeLogFileName)
        
    def test_getVersionFromChangelogFile(self):
        changeLogFileName = "dummyCHANGELOG"
        self._writeChangeLogFile(changeLogFileName)
        obsVer = CheckerUtils.getVersionFromChangelogFile(changeLogFileName)
        expVer = "1.3.6"
        
        self.assertEquals(obsVer, expVer)
        
        os.remove(changeLogFileName)
        
    def test_checkConfigVersion_check_exception_message(self):
        changeLogFileName = "dummyCHANGELOG"
        self._writeChangeLogFile(changeLogFileName)
        self._writeConfigFile("repet_version: 1.3.5\n")
        
        changeLogFileHandler = open(changeLogFileName, "r")
        configFileHandler = open(self.configFileName, "r")
        config = ConfigParser.ConfigParser()
        config.readfp(configFileHandler)
        
        checkerExceptionInstance = None
        try:
            CheckerUtils.checkConfigVersion(changeLogFileHandler, config)
        except CheckerException, e:
            checkerExceptionInstance = e
        
        expMessage = "*** Error: wrong config file version. Expected version num is 1.3.6 but actual in config file is 1.3.5"
        obsMessage = checkerExceptionInstance.msg
           
        self.assertEquals(expMessage, obsMessage)
        
    def test_checkHeaders_first_header_with_question_mark(self):
        fastaFileName = "dummyFasta.fa"
        self._writeFastaFile_first_header_with_question_mark(fastaFileName)
        fastaFileHandler = open(fastaFileName,"r")
        isExceptionRaised = False
        try:
            CheckerUtils.checkHeaders(fastaFileHandler)
        except CheckerException, e:
            isExceptionRaised = True
            
        fastaFileHandler.close()
        os.remove(fastaFileName)
            
        self.assertTrue(isExceptionRaised)
        
        expMessages = ["DmelChr4_Bla?ster_Piler_0.0_Map_3"] 
        obsMessages = e.getMessages()
        
        self.assertEquals(expMessages, obsMessages)

    def test_checkHeaders_with_2_wrong_headers(self):
        fastaFileName = "dummyFasta.fa"
        self._writeFastaFile_with_2_wrong_headers(fastaFileName)
        fastaFileHandler = open(fastaFileName,"r")
        isExceptionRaised = False
        try:
            CheckerUtils.checkHeaders(fastaFileHandler)
        except CheckerException, e:
            isExceptionRaised = True
            
        fastaFileHandler.close()
        os.remove(fastaFileName)
            
        self.assertTrue(isExceptionRaised)
        
        expMessages = ["DmelChr4_Bla?ster_Piler_0.0_Map_3","DmelChr4_Bla!ster_Piler_1.0_Map_9"] 
        obsMessages = e.getMessages()
        
        self.assertEquals(expMessages, obsMessages)

    def test_checkHeaders_with_pipe(self):
        fastaFileName = "dummyFasta.fa"
        self._writeFastaFile_with_pipe(fastaFileName)
        fastaFileHandler = open(fastaFileName,"r")
        isExceptionRaised = False
        try:
            CheckerUtils.checkHeaders(fastaFileHandler)
        except CheckerException, e:
            isExceptionRaised = True
            
        fastaFileHandler.close()
        os.remove(fastaFileName)
            
        self.assertTrue(isExceptionRaised)
        
        expMessages = ["DmelC|hr4_Blas-ter_Piler_1.0_Map_9"] 
        obsMessages = e.getMessages()
        
        self.assertEquals(expMessages, obsMessages)

    def test_checkHeaders_with_equal(self):
        fastaFileName = "dummyFasta.fa"
        self._writeFastaFile_with_equal(fastaFileName)
        fastaFileHandler = open(fastaFileName,"r")
        isExceptionRaised = False
        try:
            CheckerUtils.checkHeaders(fastaFileHandler)
        except CheckerException, e:
            isExceptionRaised = True
            
        fastaFileHandler.close()
        os.remove(fastaFileName)
            
        self.assertTrue(isExceptionRaised)
        
        expMessages = ["DmelC:hr4_Blas=ter_Piler_1.0_Map_9"] 
        obsMessages = e.getMessages()
        
        self.assertEquals(expMessages, obsMessages)

    def test_checkHeaders_all_headers_ok(self):
        fastaFileName = "dummyFasta.fa"
        self._writeFastaFile_all_headers_ok(fastaFileName)
        fastaFileHandler = open(fastaFileName,"r")
        isExceptionRaised = False
        try:
            CheckerUtils.checkHeaders(fastaFileHandler)
        except CheckerException:
            isExceptionRaised = True
            
        fastaFileHandler.close()
        os.remove(fastaFileName)
            
        self.assertFalse(isExceptionRaised)

    def _writeFastaFile_first_header_with_question_mark(self, fastaFileName):    
        fastaFileHandler = open(fastaFileName, "w")
        fastaFileHandler.write(">DmelChr4_Bla?ster_Piler_0.0_Map_3\n")
        fastaFileHandler.write("ACCAAAGACACTAGAATAACAAGATGCGTAACGCCATACGATTTTTTGGCACACTATTTT\n")
        fastaFileHandler.write("TTCGCCGTGGCTCTAGAGGTGGCTCCAGGCTCTCTCGAATTTTTGTTAGAGAGCGAGAGA\n")
        fastaFileHandler.write("GCTGAGAGCGCTACAGCGAACAGCTCTTTTCTACACATAAAGTGATAGCAGACAACTGTA\n")
        fastaFileHandler.write("TGTGTGCACACGTGTGCTCATGCATTGTAAATTTGACAAAATATGCCCTTCACCTTCAAA\n")
        fastaFileHandler.write(">DmelChr4_Blaster_Piler_1.0_Map_9\n")
        fastaFileHandler.write("AGTTTAAAAACCAAAGACACTAGAATAACAAGATGCGTAACGGCCATACATTGGTTTGGC\n")
        fastaFileHandler.write("ACTATGCAGCCACTTTTTTGGTGACGGCCAAAATTACTCTCTTTCCGCTCACTCCCGCTG\n")
        fastaFileHandler.write("AGAGCGTAAGAAATCTAAAAATATAATTTGCTTGCTTGTGTGAGTAAAAACAAGAGACGA\n")
        fastaFileHandler.write("GAACGCGTATAAGTGTGCGTGTTGTGCTAGAAGACGATTTTCGGGACCGAAATCAATTCT\n")
        fastaFileHandler.write("GATCGAAGAAACGAATTTACATGGTACATATTAGGGTAGTTTTTGCCAATTTCCTAGCAA\n")
        fastaFileHandler.close()
    
    def _writeFastaFile_with_2_wrong_headers(self, fastaFileName):    
        fastaFileHandler = open(fastaFileName, "w")
        fastaFileHandler.write(">DmelChr4_Bla?ster_Piler_0.0_Map_3\n")
        fastaFileHandler.write("ACCAAAGACACTAGAATAACAAGATGCGTAACGCCATACGATTTTTTGGCACACTATTTT\n")
        fastaFileHandler.write("TTCGCCGTGGCTCTAGAGGTGGCTCCAGGCTCTCTCGAATTTTTGTTAGAGAGCGAGAGA\n")
        fastaFileHandler.write("GCTGAGAGCGCTACAGCGAACAGCTCTTTTCTACACATAAAGTGATAGCAGACAACTGTA\n")
        fastaFileHandler.write("TGTGTGCACACGTGTGCTCATGCATTGTAAATTTGACAAAATATGCCCTTCACCTTCAAA\n")
        fastaFileHandler.write(">DmelChr4_Bla!ster_Piler_1.0_Map_9\n")
        fastaFileHandler.write("AGTTTAAAAACCAAAGACACTAGAATAACAAGATGCGTAACGGCCATACATTGGTTTGGC\n")
        fastaFileHandler.write("ACTATGCAGCCACTTTTTTGGTGACGGCCAAAATTACTCTCTTTCCGCTCACTCCCGCTG\n")
        fastaFileHandler.write("AGAGCGTAAGAAATCTAAAAATATAATTTGCTTGCTTGTGTGAGTAAAAACAAGAGACGA\n")
        fastaFileHandler.write("GAACGCGTATAAGTGTGCGTGTTGTGCTAGAAGACGATTTTCGGGACCGAAATCAATTCT\n")
        fastaFileHandler.write("GATCGAAGAAACGAATTTACATGGTACATATTAGGGTAGTTTTTGCCAATTTCCTAGCAA\n")
        fastaFileHandler.close()
    
    def _writeFastaFile_all_headers_ok(self, fastaFileName):    
        fastaFileHandler = open(fastaFileName, "w")
        fastaFileHandler.write(">DmelChr4_Blaster_Piler_0.0_Map_3\n")
        fastaFileHandler.write("ACCAAAGACACTAGAATAACAAGATGCGTAACGCCATACGATTTTTTGGCACACTATTTT\n")
        fastaFileHandler.write("TTCGCCGTGGCTCTAGAGGTGGCTCCAGGCTCTCTCGAATTTTTGTTAGAGAGCGAGAGA\n")
        fastaFileHandler.write("GCTGAGAGCGCTACAGCGAACAGCTCTTTTCTACACATAAAGTGATAGCAGACAACTGTA\n")
        fastaFileHandler.write("TGTGTGCACACGTGTGCTCATGCATTGTAAATTTGACAAAATATGCCCTTCACCTTCAAA\n")
        fastaFileHandler.write(">DmelC:hr4_Blas-ter_Piler_1.0_Map_9\n")
        fastaFileHandler.write("AGTTTAAAAACCAAAGACACTAGAATAACAAGATGCGTAACGGCCATACATTGGTTTGGC\n")
        fastaFileHandler.write("ACTATGCAGCCACTTTTTTGGTGACGGCCAAAATTACTCTCTTTCCGCTCACTCCCGCTG\n")
        fastaFileHandler.write("AGAGCGTAAGAAATCTAAAAATATAATTTGCTTGCTTGTGTGAGTAAAAACAAGAGACGA\n")
        fastaFileHandler.write("GAACGCGTATAAGTGTGCGTGTTGTGCTAGAAGACGATTTTCGGGACCGAAATCAATTCT\n")
        fastaFileHandler.write("GATCGAAGAAACGAATTTACATGGTACATATTAGGGTAGTTTTTGCCAATTTCCTAGCAA\n")
        fastaFileHandler.close()
    
    def _writeFastaFile_with_pipe(self, fastaFileName):    
        fastaFileHandler = open(fastaFileName, "w")
        fastaFileHandler.write(">DmelChr4_Blaster_Piler_0.0_Map_3\n")
        fastaFileHandler.write("ACCAAAGACACTAGAATAACAAGATGCGTAACGCCATACGATTTTTTGGCACACTATTTT\n")
        fastaFileHandler.write("TTCGCCGTGGCTCTAGAGGTGGCTCCAGGCTCTCTCGAATTTTTGTTAGAGAGCGAGAGA\n")
        fastaFileHandler.write("GCTGAGAGCGCTACAGCGAACAGCTCTTTTCTACACATAAAGTGATAGCAGACAACTGTA\n")
        fastaFileHandler.write("TGTGTGCACACGTGTGCTCATGCATTGTAAATTTGACAAAATATGCCCTTCACCTTCAAA\n")
        fastaFileHandler.write(">DmelC|hr4_Blas-ter_Piler_1.0_Map_9\n")
        fastaFileHandler.write("AGTTTAAAAACCAAAGACACTAGAATAACAAGATGCGTAACGGCCATACATTGGTTTGGC\n")
        fastaFileHandler.write("ACTATGCAGCCACTTTTTTGGTGACGGCCAAAATTACTCTCTTTCCGCTCACTCCCGCTG\n")
        fastaFileHandler.write("AGAGCGTAAGAAATCTAAAAATATAATTTGCTTGCTTGTGTGAGTAAAAACAAGAGACGA\n")
        fastaFileHandler.write("GAACGCGTATAAGTGTGCGTGTTGTGCTAGAAGACGATTTTCGGGACCGAAATCAATTCT\n")
        fastaFileHandler.write("GATCGAAGAAACGAATTTACATGGTACATATTAGGGTAGTTTTTGCCAATTTCCTAGCAA\n")
        fastaFileHandler.close()
    
    def _writeFastaFile_with_equal(self, fastaFileName):    
        fastaFileHandler = open(fastaFileName, "w")
        fastaFileHandler.write(">DmelChr4_Blaster_Piler_0.0_Map_3\n")
        fastaFileHandler.write("ACCAAAGACACTAGAATAACAAGATGCGTAACGCCATACGATTTTTTGGCACACTATTTT\n")
        fastaFileHandler.write("TTCGCCGTGGCTCTAGAGGTGGCTCCAGGCTCTCTCGAATTTTTGTTAGAGAGCGAGAGA\n")
        fastaFileHandler.write("GCTGAGAGCGCTACAGCGAACAGCTCTTTTCTACACATAAAGTGATAGCAGACAACTGTA\n")
        fastaFileHandler.write("TGTGTGCACACGTGTGCTCATGCATTGTAAATTTGACAAAATATGCCCTTCACCTTCAAA\n")
        fastaFileHandler.write(">DmelC:hr4_Blas=ter_Piler_1.0_Map_9\n")
        fastaFileHandler.write("AGTTTAAAAACCAAAGACACTAGAATAACAAGATGCGTAACGGCCATACATTGGTTTGGC\n")
        fastaFileHandler.write("ACTATGCAGCCACTTTTTTGGTGACGGCCAAAATTACTCTCTTTCCGCTCACTCCCGCTG\n")
        fastaFileHandler.write("AGAGCGTAAGAAATCTAAAAATATAATTTGCTTGCTTGTGTGAGTAAAAACAAGAGACGA\n")
        fastaFileHandler.write("GAACGCGTATAAGTGTGCGTGTTGTGCTAGAAGACGATTTTCGGGACCGAAATCAATTCT\n")
        fastaFileHandler.write("GATCGAAGAAACGAATTTACATGGTACATATTAGGGTAGTTTTTGCCAATTTCCTAGCAA\n")
        fastaFileHandler.close()

    def _writeChangeLogFile(self, changeLogFileName ):
        changeLogFileHandler = open(changeLogFileName, "w")
        changeLogFileHandler.write("ChangeLog of REPET\n")
        changeLogFileHandler.write("\n")
        changeLogFileHandler.write("\n")
        changeLogFileHandler.write("\n")
        changeLogFileHandler.write("REPET release 1.3.6\n")
        changeLogFileHandler.write("(release date XX/XX/2010)\n")
        changeLogFileHandler.write("\n")
        changeLogFileHandler.close()

    def _writeConfigFile(self, lineVersion):
        configFileHandler = open(self.configFileName, "w")
        configFileHandler.write("[repet_env]\n")
        configFileHandler.write(lineVersion)
        configFileHandler.write("repet_host: <your_MySQL_host>\n")
        configFileHandler.close()
        
        
test_suite = unittest.TestSuite()
test_suite.addTest( unittest.makeSuite( Test_CheckerUtils ) )
if __name__ == "__main__":
    unittest.TextTestRunner(verbosity=2).run( test_suite )
