# Copyright INRA (Institut National de la Recherche Agronomique)
# http://www.inra.fr
# http://urgi.versailles.inra.fr
#
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software.  You can  use, 
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info". 
#
# As a counterpart to the access to the source code and  rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty  and the software's author,  the holder of the
# economic rights,  and the successive licensors  have only  limited
# liability. 
#
# In this respect, the user's attention is drawn to the risks associated
# with loading,  using,  modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean  that it is complicated to manipulate,  and  that  also
# therefore means  that it is reserved for developers  and  experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or 
# data to be ensured and,  more generally, to use and operate it in the 
# same conditions as regards security. 
#
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.

import unittest
import os
import time
from commons.core.sql.TableBinPathAdaptator import TableBinPathAdaptator
from commons.core.coord.Path import Path
from commons.core.coord.Set import Set
from commons.core.sql.DbFactory import DbFactory

class Test_TableBinPathAdaptator( unittest.TestCase ):
    
    def setUp( self ):
        self._uniqId = "%s_%s" % (time.strftime("%Y%m%d%H%M%S") , os.getpid())
        self._db = DbFactory.createInstance()
        self._table = "dummyPathTable_%s" % self._uniqId
        self._table_idx = "dummyPathTable_%s_idx" % self._uniqId
        
    def tearDown( self ):
        self._db.dropTable(self._table)
        self._db.dropTable(self._table_idx)
        self._db.close()
 
    #TODO: strand ?!? How does it work ?
    def test_insert_QryRevSbjDir( self ):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("1", "chr1", "250", "100", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        tuple = ("4", "chr5", "140", "251", "TE5", "140", "251", "2e-14", "14", "73.1")
        p4 = Path()
        p4.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        self._tpA.insert(p4)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (1, "chr1", 100, 250, "TE1", 17, 11, 1e-18, 20, 87.4),
                        (2, "chr1", 15, 30, "TE2", 10, 13, 5e-24, 34, 93.1),
                        (4, "chr5", 140, 251, "TE5", 140, 251, 2e-14, 14, 73.1),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),
                        (1, 1000000, "chr1", 100, 250, 1),
                        (2, 1000000, "chr1", 15, 30, 1),
                        (4, 1000000, "chr5", 140, 251, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_getPathListOverlappingQueryCoord_one_included( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("6\tchr1\t950\t1010\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()

        tuple = ("6", "chr1", "950", "1010", "TE2", "11", "17", "1e-20", "30", "90.2")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        
        lObs = self._tpA.getPathListOverlappingQueryCoord( "chr1", 900, 1010 )
        self.assertEquals(1, len(lObs))
        lExp = [p1]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )
        
    def test_getPathListOverlappingQueryCoord_two_overlapped( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("6\tchr1\t950\t1500\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("7\tchr1\t750\t1000\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()

        tuple = ("6", "chr1", "950", "1500", "TE2", "11", "17", "1e-20", "30", "90.2")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("7", "chr1", "750", "1000", "TE2", "11", "17", "1e-20", "30", "90.2")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        
        lObs = self._tpA.getPathListOverlappingQueryCoord( "chr1", 900, 1010 )
        self.assertEquals(2, len(lObs))
        lExp = [p1, p2]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )
        
    def test_getPathListOverlappingQueryCoord_two_not_overlapped_and_not_included( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("6\tchr1\t1050\t1500\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("7\tchr1\t750\t800\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()

        tuple = ("6", "chr1", "1050", "1500", "TE2", "11", "17", "1e-20", "30", "90.2")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("7", "chr1", "750", "800", "TE2", "11", "17", "1e-20", "30", "90.2")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        
        lObs = self._tpA.getPathListOverlappingQueryCoord( "chr1", 900, 1010 )
        self.assertEquals(0, len(lObs))
        lExp = []
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )
        
    def test_getPathListOverlappingQueryCoord_one_verlapping_and_others_chained( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("6\tchr1\t900\t1010\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("6\tchr1\t1020\t1030\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("7\tchr1\t950\t999\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("7\tchr1\t1020\t1030\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("7\tchr5\t8000\t15000\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()

        tuple = ("6", "chr1", "900", "1010", "TE2", "11", "17", "1e-20", "30", "90.2")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("6", "chr1", "1020", "1030", "TE2", "10", "13", "1e-20", "30", "90.2")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("7", "chr1", "850", "999", "TE2", "11", "17", "1e-20", "30", "90.2")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        tuple = ("7", "chr1", "1020", "1030", "TE2", "11", "17", "1e-20", "30", "90.2")
        p4 = Path()
        p4.setFromTuple(tuple)
        
        tuple = ("7", "chr5", "8000", "15000", "TE2", "11", "17", "1e-20", "30", "90.2")
        p5 = Path()
        p5.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        
        lObs = self._tpA.getPathListOverlappingQueryCoord( "chr1", 1000, 1010 )
        self.assertEquals(1, len(lObs))
        lExp = [p1]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )
        
    def test_getChainListOverlappingQueryCoord_with_all_path_strictly_included_in_the_given_region(self):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t10\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("3\tchr1\t8\t13\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("4\tchr1\t11\t15\tTE2\t15\t19\t1e-10\t25\t80.2\n")
        pathF.write("5\tchr1\t14\t19\tTE1\t1\t6\t1e-15\t45\t98.4\n")
        pathF.close()

        tuple = ("1", "chr1", "1", "10", "TE2", "11", "17", "1e-20", "30", "90.2")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "2", "9", "TE2", "10", "13", "1e-20", "30", "90.2")
        p2 = Path()
        p2.setFromTuple(tuple)

        tuple = ("3", "chr1", "8", "13", "TE2", "11", "17", "1e-20", "30", "90.2")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        tuple = ("4", "chr1", "11", "15", "TE2", "15", "19", "1e-10", "25", "80.2")
        p4 = Path()
        p4.setFromTuple(tuple)
        
        tuple = ("5", "chr1", "14", "19", "TE1", "1", "6", "1e-15", "45", "98.4")
        p5 = Path()
        p5.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        lObs = self._tpA.getChainListOverlappingQueryCoord( "chr1", 1, 20 )
        self.assertEquals(5, len(lObs))
        
        lExp = [p1, p2, p3, p4, p5]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )
        
    def test_getChainListOverlappingQueryCoord_with_2_path_overlapping_the_given_region(self):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t20\tTE1\t11\t17\t1e-18\t20\t87.4\n")
        pathF.write("2\tchr1\t10\t30\tTE2\t10\t13\t5e-24\t34\t93.1\n")
        pathF.close()

        tuple = ("1", "chr1", "1", "20", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "10", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        lObs = self._tpA.getChainListOverlappingQueryCoord( "chr1", 12, 18 )
        self.assertEquals(2, len(lObs))
        
        lExp = [p1, p2]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )
        
    def test_getChainListOverlappingQueryCoord_without_path_overlapping_the_given_region(self):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t1\t20\tTE1\t11\t17\t1e-18\t20\t87.4\n")
        pathF.write("2\tchr1\t10\t30\tTE2\t10\t13\t5e-24\t34\t93.1\n")
        pathF.write("3\tchr5\t45\t50\tTE2\t10\t13\t5e-24\t34\t93.1\n")
        pathF.close()

        tuple = ("1", "chr1", "1", "20", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "10", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr5", "45", "50", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        lObs = self._tpA.getChainListOverlappingQueryCoord( "chr1", 40, 50 )
        self.assertEquals(0, len(lObs))
        
        lExp = []
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )
        
    def test_getChainListOverlappingQueryCoord_with_inverse_coord(self):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t2000\t1010\tTE2\t17\t11\t1e-20\t30\t90.2\n")
        pathF.write("2\tchr1\t5000\t3030\tTE2\t13\t10\t1e-20\t30\t90.2\n")
        pathF.close()
        
        tuple = ("1", "chr1", "2000", "1010", "TE2", "17", "11", "1e-20", "30", "90.2")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "5000", "3030", "TE2", "13", "10", "1e-20", "30", "90.2")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        
        lObs = self._tpA.getChainListOverlappingQueryCoord( "chr1", 1000, 1500 )
        self.assertEquals(1, len(lObs))
        lExp = [p1]
        self.assertEquals(lExp, lObs)
        
        lObs = self._tpA.getChainListOverlappingQueryCoord( "chr1", 4000, 4510 )
        self.assertEquals(1, len(lObs))
        lExp = [p2]
        self.assertEquals(lExp, lObs)
        
        lObs = self._tpA.getChainListOverlappingQueryCoord( "chr1", 1000, 4510 )
        self.assertEquals(2, len(lObs))
        lExp = [p1, p2]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )
        
    def test_getChainListOverlappingQueryCoord_with_chain_id_and_coord( self ):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("6\tchr1\t900\t1010\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("6\tchr1\t1020\t1030\tTE2\t10\t13\t1e-20\t30\t90.2\n")
        pathF.write("7\tchr1\t950\t999\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("7\tchr1\t1020\t1030\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.write("7\tchr5\t8000\t15000\tTE2\t11\t17\t1e-20\t30\t90.2\n")
        pathF.close()

        tuple = ("6", "chr1", "900", "1010", "TE2", "11", "17", "1e-20", "30", "90.2")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("6", "chr1", "1020", "1030", "TE2", "10", "13", "1e-20", "30", "90.2")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("7", "chr1", "950", "999", "TE2", "11", "17", "1e-20", "30", "90.2")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        tuple = ("7", "chr1", "1020", "1030", "TE2", "11", "17", "1e-20", "30", "90.2")
        p4 = Path()
        p4.setFromTuple(tuple)
        
        tuple = ("7", "chr5", "8000", "15000", "TE2", "11", "17", "1e-20", "30", "90.2")
        p5 = Path()
        p5.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        lObs = self._tpA.getChainListOverlappingQueryCoord( "chr1", 1000, 1010 )
        self.assertEquals(5, len(lObs))
        lExp = [p1, p2, p3, p4, p5]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )

    def test_getPathListIncludedInQueryCoord_all_included(self):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t10\t20\tTE1\t11\t17\t1e-18\t20\t87.4\n")
        pathF.write("2\tchr1\t20\t30\tTE2\t10\t13\t5e-24\t34\t93.1\n")
        pathF.close()

        tuple = ("1", "chr1", "10", "20", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "20", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        lObs = self._tpA.getPathListIncludedInQueryCoord( "chr1", 1, 40 )
        self.assertEquals(2, len(lObs))
        
        lExp = [p1, p2]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )

    def test_getPathListIncludedInQueryCoord_all_not_included(self):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t10\t20\tTE1\t11\t17\t1e-18\t20\t87.4\n")
        pathF.write("2\tchr1\t20\t30\tTE2\t10\t13\t5e-24\t34\t93.1\n")
        pathF.write("3\tchr5\t55\t60\tTE2\t10\t13\t5e-24\t34\t93.1\n")
        pathF.close()

        tuple = ("1", "chr1", "10", "20", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "20", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr2", "55", "60", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        lObs = self._tpA.getPathListIncludedInQueryCoord( "chr1", 50, 70 )
        self.assertEquals(0, len(lObs))
        
        lExp = []
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )

    def test_getPathListIncludedInQueryCoord_all_overlapping(self):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t10\t25\tTE1\t11\t17\t1e-18\t20\t87.4\n")
        pathF.write("2\tchr1\t15\t30\tTE2\t10\t13\t5e-24\t34\t93.1\n")
        pathF.close()

        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        lObs = self._tpA.getPathListIncludedInQueryCoord( "chr1", 13, 22 )
        self.assertEquals(0, len(lObs))
        
        lExp = []
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )

    def test_getPathListIncludedInQueryCoord_with_one_included_and_one_overlapping(self):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t10\t25\tTE1\t11\t17\t1e-18\t20\t87.4\n")
        pathF.write("2\tchr1\t15\t30\tTE2\t10\t13\t5e-24\t34\t93.1\n")
        pathF.close()

        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        lObs = self._tpA.getPathListIncludedInQueryCoord( "chr1", 9, 27 )
        self.assertEquals(1, len(lObs))
        lExp = [p1]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )

    def test_getPathListIncludedInQueryCoord_with_one_included_and_two_chained(self):
        pathFileName = "dummyPathFile_%s" % ( self._uniqId )
        pathF = open( pathFileName, "w" )
        pathF.write("1\tchr1\t10\t25\tTE1\t11\t17\t1e-18\t20\t87.4\n")
        pathF.write("1\tchr1\t100\t250\tTE1\t11\t17\t1e-18\t20\t87.4\n")
        pathF.write("2\tchr1\t15\t30\tTE2\t10\t13\t5e-24\t34\t93.1\n")
        pathF.close()

        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("1", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path", pathFileName )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        lObs = self._tpA.getPathListIncludedInQueryCoord( "chr1", 9, 27 )
        self.assertEquals(1, len(lObs))
        lExp = [p1]
        self.assertEquals(lExp, lObs)
        
        os.remove( pathFileName )
        
    def test_deleteFromId_with_correct_ID(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        self._tpA.deleteFromId(3)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (2, "chr1", 100, 250, "TE1", 11, 17, 1e-18, 20, 87.4),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),
                        (2, 1000000, "chr1", 100, 250, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_deleteFromId_with_not_exist_ID(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        self._tpA.deleteFromId(4)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (2, "chr1", 100, 250, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (3, "chr1", 15, 30, "TE2", 10, 13, 5e-24, 34, 93.1),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),
                        (2, 1000000, "chr1", 100, 250, 1),
                        (3, 1000000, "chr1", 15, 30, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_deleteFromId_with_multiple_ID(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("2", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        self._tpA.deleteFromId(2)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_deleteFromIdList(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        lNumToRemove = [2, 3]
        self._tpA.deleteFromIdList(lNumToRemove)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_deleteFromIdList_with_empty_list(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        lNumToRemove = []
        self._tpA.deleteFromIdList(lNumToRemove)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (2, "chr1", 100, 250, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (3, "chr1", 15, 30, "TE2", 10, 13, 5e-24, 34, 93.1),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),
                        (2, 1000000, "chr1", 100, 250, 1),
                        (3, 1000000, "chr1", 15, 30, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_deleteFromIdList_with_list_of_existing_and_not_existing_ID(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        lNumToRemove = [3, 4]
        self._tpA.deleteFromIdList(lNumToRemove)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (2, "chr1", 100, 250, "TE1", 11, 17, 1e-18, 20, 87.4),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),
                        (2, 1000000, "chr1", 100, 250, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_deleteFromIdList_with_multiple_ID_on_BinPathTable(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("3", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        lNumToRemove = [3]
        self._tpA.deleteFromIdList(lNumToRemove)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_joinTwoPaths_with_min_and_max_existing_IDs(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        expNewId = 1
        obsNewId = self._tpA.joinTwoPaths(1, 2)
        
        self.assertEquals(expNewId, obsNewId)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (1, "chr1", 100, 250, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (3, "chr1", 15, 30, "TE2", 10, 13, 5e-24, 34, 93.1),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),
                        (1, 1000000, "chr1", 100, 250, 1),
                        (3, 1000000, "chr1", 15, 30, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_joinTwoPaths_with_min_ID_not_existing_and_max_ID_existing(self):
        tuple = ("4", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("5", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("6", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        expNewId = 1
        obsNewId = self._tpA.joinTwoPaths(1, 5)
        
        self.assertEquals(expNewId, obsNewId)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((4, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (1, "chr1", 100, 250, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (6, "chr1", 15, 30, "TE2", 10, 13, 5e-24, 34, 93.1),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((4, 1000000, "chr1", 10, 25, 1),
                        (1, 1000000, "chr1", 100, 250, 1),
                        (6, 1000000, "chr1", 15, 30, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_joinTwoPaths_with_min_ID_existing_and_max_ID_not_existing(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        expNewId = 1
        obsNewId = self._tpA.joinTwoPaths(1, 5)
        
        self.assertEquals(expNewId, obsNewId)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (2, "chr1", 100, 250, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (3, "chr1", 15, 30, "TE2", 10, 13, 5e-24, 34, 93.1),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),
                        (2, 1000000, "chr1", 100, 250, 1),
                        (3, 1000000, "chr1", 15, 30, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_joinTwoPaths_with_min_and_max_not_existing_IDs(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        expNewId = 4
        obsNewId = self._tpA.joinTwoPaths(4, 5)
        
        self.assertEquals(expNewId, obsNewId)
        
        sqlCmd = "SELECT * FROM %s" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, "chr1", 10, 25, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (2, "chr1", 100, 250, "TE1", 11, 17, 1e-18, 20, 87.4),
                        (3, "chr1", 15, 30, "TE2", 10, 13, 5e-24, 34, 93.1),)
        self.assertEquals(expPathTuple, obsPathTuple)

        sqlCmd = "SELECT * FROM %s_idx" % ( self._table )
        self._db.execute( sqlCmd )
        obsPathTuple = self._db.cursor.fetchall()
        expPathTuple = ((1, 1000000, "chr1", 10, 25, 1),
                        (2, 1000000, "chr1", 100, 250, 1),
                        (3, 1000000, "chr1", 15, 30, 1),)
        self.assertEquals(expPathTuple, obsPathTuple)
        
    def test_getNewId(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        expNewId = 4
        obsNewId = self._tpA.getNewId()
        
        self.assertEquals(expNewId, obsNewId)
        
    def test_getNewId_with_empty_path_table(self):
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        
        expNewId = 1
        obsNewId = self._tpA.getNewId()
        
        self.assertEquals(expNewId, obsNewId)
        
    def test_getSetListIncludedInQueryCoord_one_included(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        
        s2 = Set()
        s2.setFromTuple(("2","TE1","chr1","100","250"))
        expLSet = [s2]
        obsLSet = self._tpA.getSetListIncludedInQueryCoord('chr1', 95, 300)
        
        self.assertEquals(expLSet, obsLSet)
        
    def test_getSetListIncludedInQueryCoord_one_overlapping(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        
        expLSet = []
        obsLSet = self._tpA.getSetListIncludedInQueryCoord('chr1', 150, 200)
        
        self.assertEquals(expLSet, obsLSet)
        
    def test_getSetListIncludedInQueryCoord_with_no_result(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        
        expLSet = []
        obsLSet = self._tpA.getSetListIncludedInQueryCoord('chr1', 5000, 6000)
        
        self.assertEquals(expLSet, obsLSet)
        
    def test_getSetListIncludedInQueryCoord_one_included_and_two_chain(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)

        tuple = ("2", "chr1", "1000", "2500", "TE1", "11", "17", "1e-18", "20", "87.4")
        p3 = Path()
        p3.setFromTuple(tuple)

        tuple = ("3", "chr1", "50", "150", "TE1", "11", "17", "1e-18", "20", "87.4")
        p4 = Path()
        p4.setFromTuple(tuple)
        
        tuple = ("4", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p5 = Path()
        p5.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        self._tpA.insert(p4)
        self._tpA.insert(p5)
        
        s2 = Set()
        s2.setFromTuple(("2","TE1","chr1","100","250"))
        expLSet = [s2]
        obsLSet = self._tpA.getSetListIncludedInQueryCoord('chr1', 95, 300)
        
        self.assertEquals(expLSet, obsLSet)
        
    def test_getSetListOverlappingQueryCoord_one_included(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        
        s2 = Set()
        s2.setFromTuple(("2","TE1","chr1","100","250"))
        expLSet = [s2]
        obsLSet = self._tpA.getSetListOverlappingQueryCoord('chr1', 95, 300)
        
        self.assertEquals(expLSet, obsLSet)
        
    def test_getSetListOverlappingQueryCoord_one_overlapping(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        
        s2 = Set()
        s2.setFromTuple(("2","TE1","chr1","100","250"))
        expLSet = [s2]
        obsLSet = self._tpA.getSetListOverlappingQueryCoord('chr1', 150, 200)
        
        self.assertEquals(expLSet, obsLSet)
        
    def test_getSetListOverlappingQueryCoord_with_no_result(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        
        expLSet = []
        obsLSet = self._tpA.getSetListOverlappingQueryCoord('chr1', 5000, 6000)
        
        self.assertEquals(expLSet, obsLSet)
        
    def test_getSetListOverlappingQueryCoord_one_included_and_two_chain(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)

        tuple = ("2", "chr1", "1000", "2500", "TE1", "11", "17", "1e-18", "20", "87.4")
        p3 = Path()
        p3.setFromTuple(tuple)

        tuple = ("3", "chr1", "50", "150", "TE1", "11", "17", "1e-18", "20", "87.4")
        p4 = Path()
        p4.setFromTuple(tuple)
        
        tuple = ("4", "chr1", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p5 = Path()
        p5.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        self._tpA.insert(p4)
        self._tpA.insert(p5)
        
        s2 = Set()
        s2.setFromTuple(("2","TE1","chr1","100","250"))
        s4 = Set()
        s4.setFromTuple(("3","TE1","chr1","50","150"))
        expLSet = [s2, s4]
        obsLSet = self._tpA.getSetListOverlappingQueryCoord('chr1', 95, 300)
        
        self.assertEquals(expLSet, obsLSet)
        
    def test_getIdList( self ):
        p1 = Path()
        p1.setFromString( "1\tchr1\t1\t10\tTE1\t11\t17\t1e-20\t30\t90.2\n" )
        p2 = Path()
        p2.setFromString( "2\tchr1\t2\t9\tTE2\t10\t13\t1e-20\t30\t90.2\n" )
        p3 = Path()
        p3.setFromString( "2\tchr1\t12\t19\tTE2\t15\t22\t1e-10\t40\t94.2\n" )
        p4 = Path()
        p4.setFromString( "3\tchr2\t8\t13\tTE1\t11\t17\t1e-20\t30\t90.2\n" )
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        
        lPath = [ p1, p2, p3, p4]
        self._tpA.insertList(lPath)
        
        expList = [ 1, 2, 3 ]
        obsList = self._tpA.getIdList()
        
        self.assertEqual( expList, obsList )
        
    def test_getQueryList(self):
        tuple = ("1", "chr1", "10", "25", "TE1", "11", "17", "1e-18", "20", "87.4")
        p1 = Path()
        p1.setFromTuple(tuple)

        tuple = ("2", "chr1", "100", "250", "TE1", "11", "17", "1e-18", "20", "87.4")
        p2 = Path()
        p2.setFromTuple(tuple)
        
        tuple = ("3", "chr2", "15", "30", "TE2", "10", "13", "5e-24", "34", "93.1")
        p3 = Path()
        p3.setFromTuple(tuple)
        
        self._db.createTable( self._table, "path" )
        self._db.createBinPathTable(self._table, True)
        self._tpA = TableBinPathAdaptator( self._db, self._table )
        self._tpA.insert(p1)
        self._tpA.insert(p2)
        self._tpA.insert(p3)
        
        expList = [ "chr1", "chr2" ]
        obsList = self._tpA.getQueryList()
        self.assertEqual( expList, obsList )

test_suite = unittest.TestSuite()
test_suite.addTest( unittest.makeSuite( Test_TableBinPathAdaptator ) )
if __name__ == '__main__':
    unittest.TextTestRunner(verbosity=2).run( test_suite )
