#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import re
import sys
from SMART.Java.Python.structure.Interval import Interval
from commons.core.parsing.TranscriptListParser import TranscriptListParser
from SMART.Java.Python.structure.Transcript import Transcript


class BedParser(TranscriptListParser):
    """A class that parses a BED file and create a transcript list"""


    def __init__(self, fileName, verbosity = 0):
        self.title = None
        TranscriptListParser.__init__(self, fileName, verbosity)


#    def __del__(self):
#        super(BedParser, self).__del__()


    def getFileFormats():
        return ["bed"]
    getFileFormats = staticmethod(getFileFormats)


    def skipFirstLines(self):
        mark = self.handle.tell()
        line = self.handle.readline()
        line = line.strip()
        m = re.search(r"^\s*track\s+name\s*=\s*(\S+)\s+", line)
        if m != None:
            self.title = m.group(1)
            self.currentLineNb += 1
        else:
            self.handle.seek(mark)
        return
            



    def parseLine(self, line):
        m = re.search(r"^\s*(\S+)\t+(\d+)\t+(\d+)\s*$", line)
        if m != None:
            transcript = Transcript()
            transcript.setChromosome(m.group(1))
            transcript.setStart(min(int(m.group(2)), int(m.group(3))-1))
            transcript.setEnd(max(int(m.group(2)), int(m.group(3))-1))
            transcript.setName("Unnamed")
            transcript.setDirection(1)
            return transcript

        m = re.search(r"^\s*(\S+)\t+(\d+)\t+(\d+)\t+([^\t]+)\s*$", line)
        if m != None:
            transcript = Transcript()
            transcript.setChromosome(m.group(1))
            transcript.setStart(min(int(m.group(2)), int(m.group(3))-1))
            transcript.setEnd(max(int(m.group(2)), int(m.group(3))-1))
            transcript.setName(m.group(4))
            transcript.setDirection(1)
            return transcript

        m = re.search(r"^\s*(\S+)\t+(\d+)\t+(\d+)\t+([^\t]+)\t+\d+\.?\d*\s*$", line)
        if m != None:
            transcript = Transcript()
            transcript.setChromosome(m.group(1))
            transcript.setStart(min(int(m.group(2)), int(m.group(3))-1))
            transcript.setEnd(max(int(m.group(2)), int(m.group(3))-1))
            transcript.setName(m.group(4))
            transcript.setDirection(1)
            return transcript

        m = re.search(r"^\s*(\S+)\t+(\d+)\t+(\d+)\t+([^\t]+)\t+\d+\t+([+-])\t+\d+\t+\d+\t+0\t+(\d+)\t+(\S+)\t+(\S+)\s*$", line)
        if m == None:
            raise Exception("\nLine %d '%s' does not has a BED format." % (self.currentLineNb, line))
        transcript = Transcript()
        transcript.setChromosome(m.group(1))
        transcript.setStart(min(int(m.group(2)), int(m.group(3))-1))
        transcript.setEnd(max(int(m.group(2)), int(m.group(3))-1))
        transcript.setName(m.group(4))
        transcript.setDirection(m.group(5))
        nbExons = int(m.group(6))
        sizes = m.group(7).split(",")
        starts = m.group(8).split(",")

        # check for comment in name
        m = re.search(r"^([^\(]*)\((\S+)\)$", transcript.getName())
        if m != None:
            transcript.setName(m.group(1))
            transcript.setTagValues(m.group(2), ";", "=")
        
        # check for nb occurrences in name
        m = re.search(r"(.*)-(\d+)$", transcript.getName())
        if m != None:
            transcript.setName(m.group(1))
            transcript.setOccurrence(int(m.group(2)))

        for i in range(nbExons):
            exon = Interval(transcript)
            exon.setStart(int(starts[i])+transcript.getStart())
            exon.setEnd(transcript.getStart()+int(starts[i])+int(sizes[i])-1)
            exon.setSize(int(sizes[i]))
            transcript.addExon(exon)
            
        if transcript.exons[0].getStart() != transcript.getStart():
            sys.exit("There is something wrong with the start of transcript line '%s': transcript starts at %d whereas first exon starts at %d" % (line.strip(), transcript.start, transcript.exons[0].start))
        if transcript.exons[-1].getEnd() != transcript.getEnd():
            sys.exit("There is something wrong with the end of transcript line '%s': transcript ends at %d whereas last exon ends at %d" % (line.strip(), transcript.end, transcript.exons[-1].end))

        return transcript

