#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2012
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#

import random, os, time, shutil
from optparse import OptionParser
from commons.core.parsing.ParserChooser import ParserChooser
from SMART.Java.Python.structure.Transcript import Transcript
from SMART.Java.Python.structure.Interval import Interval
from SMART.Java.Python.ncList.NCList import NCList
from SMART.Java.Python.ncList.NCListCursor import NCListCursor
from SMART.Java.Python.ncList.NCListFilePickle import NCListFilePickle, NCListFileUnpickle
from SMART.Java.Python.ncList.FileSorter import FileSorter
from SMART.Java.Python.ncList.NCListMerger import NCListMerger
from SMART.Java.Python.misc.Progress import Progress
from SMART.Java.Python.misc.UnlimitedProgress import UnlimitedProgress
try:
   import cPickle as pickle
except:
   import pickle

class ConvertToNCList(object):
	
	def __init__(self, verbosity = 1):
		self._parsers				  = {}
		self._sortedFileNames		  = {}
		self._inputFileName		      = None
		self._outputFileName		  = None
		self._index				      = False
		self._ncLists				  = {}
		self._splittedFileNames	      = {}
		self._nbElements			  = 0
		self._nbElementsPerChromosome = {}
		self._randomNumber			  = random.randint(0, 10000)
		self._sorted                  = False
		self._verbosity			      = verbosity
		
	def setInputFileName(self, fileName, format):
		self._inputFileName = fileName
		chooser = ParserChooser(self._verbosity)
		chooser.findFormat(format)
		self._parser = chooser.getParser(fileName)
		
	def setOutputFileName(self, fileName):
		self._outputFileName = fileName
		fileNameNoExtension  = os.path.splitext(fileName)[0]
		baseName			 = "%s_%d" % (fileNameNoExtension, self._randomNumber)
		self._directory	     = "%s_files" % (baseName)
		if not os.path.exists(self._directory):
			os.makedirs(self._directory)
		self._sortedFileNames = os.path.join(self._directory, baseName)

	def setIndex(self, boolean):
		self._index = boolean

	def setSorted(self, boolean):
		self._sorted = boolean

	def sortFile(self):
		if self._verbosity > 2:
			print "%s file %s..." % ("Rewriting" if self._sorted else "Sorting", self._inputFileName)
		startTime = time.time()
		fs = FileSorter(self._parser, self._verbosity-4)
		fs.setPresorted(self._sorted)
		fs.perChromosome(True)
		fs.setOutputFileName(self._sortedFileNames)
		fs.sort()
		self._splittedFileNames	      = fs.getOutputFileNames()
		self._nbElementsPerChromosome = fs.getNbElementsPerChromosome()
		self._nbElements			  = fs.getNbElements()
		endTime = time.time()
		if self._verbosity > 2:
			print "	...done (%ds)" % (endTime - startTime)
			
	def createNCLists(self):
		self._ncLists = {}
		if self._verbosity > 2:
			print "Creating NC-list for %s..." % (self._inputFileName)
		startTime = time.time()
		for chromosome, fileName in self._splittedFileNames.iteritems():
			if self._verbosity > 3:
				print "  chromosome %s" % (chromosome)
			ncList = NCList(self._verbosity)
			if self._index:
				ncList.createIndex(True)
			ncList.setChromosome(chromosome)
			ncList.setFileName(fileName)
			ncList.setNbElements(self._nbElementsPerChromosome[chromosome])
			ncList.buildLists()
			self._ncLists[chromosome] = ncList
		endTime = time.time()
		if self._verbosity > 2:
			print "	...done (%ds)" % (endTime - startTime)

	def writeOutputFile(self):
		merger = NCListMerger(self._verbosity)
		merger.setFileName(self._outputFileName)
		merger.addIndex(self._index)
		merger.setNCLists(self._ncLists)
		merger.merge()

	def cleanFiles(self):
		shutil.rmtree(self._directory)

	def run(self):
		self.sortFile()
		self.createNCLists()
		self.writeOutputFile()
		self.cleanFiles()

	def getSortedFileNames(self):
		return self._splittedFileNames

	def getNbElements(self):
		return self._nbElements

	def getNbElementsPerChromosome(self):
		return self._nbElementsPerChromosome

	def getNCLists(self):
		return self._ncLists

	def getTmpDirectory(self):
		return self._directory


if __name__ == "__main__":
	description = "Convert To NC-List v1.0.0: Convert a mapping or transcript file into a NC-List. [Category: NC-List]"

	parser = OptionParser(description = description)
	parser.add_option("-i", "--input",	   dest="inputFileName",  action="store",					  type="string",  help="Query input file [compulsory] [format: file in transcript format given by -f]")
	parser.add_option("-f", "--format",	   dest="format",		  action="store",					  type="string",  help="format of previous file [compulsory] [format: transcript file format]")
	parser.add_option("-d", "--index",	   dest="index",		  action="store_true", default=False,				  help="create an index [default: false] [format: boolean]")
	parser.add_option("-o", "--output",	   dest="outputFileName", action="store",					  type="string",  help="Output file [compulsory] [format: output file in NCList format]")
	parser.add_option("-s", "--sorted",	   dest="sorted",	      action="store_true", default=False,	              help="input file is already sorted [format: boolean] [default: False]")
	parser.add_option("-v", "--verbosity", dest="verbosity",	  action="store",	   default=1,	  type="int",	  help="Trace level [format: int] [default: 1]")
	(options, args) = parser.parse_args()
	
	ctncl = ConvertToNCList(options.verbosity)
	ctncl.setInputFileName(options.inputFileName, options.format)
	ctncl.setOutputFileName(options.outputFileName)
	ctncl.setIndex(options.index)
	ctncl.setSorted(options.sorted)
	ctncl.run()

