#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import re
import sys
from SMART.Java.Python.structure.Interval import Interval
from SMART.Java.Python.structure.Transcript import Transcript
from commons.core.parsing.TranscriptListParser import TranscriptListParser


class GbParser(TranscriptListParser):
    """A class that parses a GBrowse file and create a transcript list"""


    def __init__(self, fileName, verbosity = 0):
        self.reference = None
        self.color         = None
        super(GbParser, self).__init__(fileName, verbosity)


    def __del__(self):
        super(GbParser, self).__del__()


    def getFileFormats():
        return ["gb", "gbrowse"]
    getFileFormats = staticmethod(getFileFormats)


    def skipFirstLines(self):
        for line in self.handle:
            self.currentLineNb += 1
            line = line.strip()
            m = re.search(r"^\s*bgcolor\s*=\s*(\S+)\s*$", line)
            if m != None:
                self.color = m.group(1)
            if line == "":
                return


    def parseLine(self, line):
        transcript = Transcript()
        # first line (reference)
        m = re.search(r"^\s*reference\s*=\s*(\S+)\s*$", line)
        if m != None:
            self.reference = m.group(1)
            for line in self.handle:
                line = line.strip()
                self.currentLineNb += 1
                break
        # second line (genomic coordinates)
        m = re.search(r"^\s*READS\s+(\S+)\s+(\S+)\s+\"([^\"]*)\"\s*$", line)
        if m == None:
            sys.exit("\nLine %d '%s' does not have a GBrowse format" % (self.currentLineNb, line))
        if self.reference == None:
            sys.exit("Cannot get reference of GBrowse line %d '%s'" % (self.currentLineNb, line))
        transcript.setChromosome(self.reference)
        transcript.setName(m.group(1))
        transcript.setComment(m.group(3))
        # exons
        exons = m.group(2).split(",")
        transcriptStart = 1000000000
        transcriptEnd = 0
        direction = 0
        for exon in exons:
            m = re.search(r"^(\d+)-(\d+)$", exon)
            if m == None:
                sys.exit("\nCannot read GBrowse exon line %d '%s'" % (self.currentLineNb, exon))
            interval = Interval()
            interval.setChromosome(transcript.chromosome)
            direction += int(m.group(2)) - int(m.group(1))
            start = min(int(m.group(1)), int(m.group(2)))
            end     = max(int(m.group(1)), int(m.group(2)))
            interval.setStart(start)
            interval.setEnd(end)
            transcriptStart = min(transcriptStart, start)
            transcriptEnd     = max(transcriptEnd, end)
            transcript.addExon(interval)
        transcript.setStart(transcriptStart)
        transcript.setEnd(transcriptEnd)
        transcript.setDirection(direction)
        for exon in transcript.getExons():
            exon.setDirection(direction)
        return transcript

