#! /usr/bin/env python
#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
"""
Clean a GTF file
"""

import shlex
from SMART.Java.Python.cleaning.TranscriptListCleaner import TranscriptListCleaner
from SMART.Java.Python.misc.Progress import Progress
from SMART.Java.Python.misc.UnlimitedProgress import UnlimitedProgress

count = {}

class ParsedLine(object):
	def __init__(self, line, cpt):
		self.line = line
		self.cpt  = cpt
		self.parse()

	def parse(self):
		self.line = self.line.strip()
		self.splittedLine = self.line.split(None, 8)
		if len(self.splittedLine) < 9:
			raise Exception("Line '%s' has less than 9 fields.  Exiting..." % (self.line))
		self.type = self.splittedLine[2]
		self.parseOptions()

	def parseOptions(self):
		self.parsedOptions = {}
		key   = None
		value = ""
		for option in shlex.split(self.splittedLine[8]):
			option = option.strip()
			if option == "": continue
			if key == None:
				key = option
			else:
				endValue = False
				if option[-1] == ";":
					endValue = True
					option.rstrip(";")
				value = "%s \"%s\"" % (value, option)
				if endValue:
					self.parsedOptions[key] = value
					if key == "transcript_id":
						self.transcriptId = value
					key   = None
					value = ""

	def export(self):
		return "%s\n" % (self.line)


class GtfCleaner(TranscriptListCleaner):

	def __init__(self, verbosity = 1):
		super(GtfCleaner, self).__init__(verbosity)
		self.acceptedTypes = ["exon"]
		self.parents	   = {}

	def getFileFormats():
		return ["gtf"]
	getFileFormats = staticmethod(getFileFormats)

	def setAcceptedTypes(self, types):
		self.acceptedTypes = types

	def parse(self):
		progress = UnlimitedProgress(100000, "Reading input file", self.verbosity)
		for cpt, line in enumerate(self.inputHandle):
			if not line or line[0] == "#": continue
			parsedLine = ParsedLine(line, cpt)
			if self.acceptedTypes == None or parsedLine.type in self.acceptedTypes:
				transcriptId = parsedLine.transcriptId
				if transcriptId not in self.parents:
					self.parents[parsedLine.transcriptId] = [parsedLine]
				else:
					self.parents[parsedLine.transcriptId].append(parsedLine)
			progress.inc()
		progress.done()

	def write(self):
		progress = Progress(len(self.parents.keys()), "Writing output file", self.verbosity)
		for parent in sorted(self.parents.keys()):
			for line in self.parents[parent]:
				self.outputHandle.write(line.export())
			progress.inc()
		progress.done()

	def _clean(self):
		self.parse()
		self.write()

