#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
from SMART.Java.Python.misc.Progress import Progress

class TranscriptListWriter(object):
    """
    An interface that writes a transcript list into a file
    @ivar fileName: name of the file 
    @type fileName: string
    @ivar handle: handle to the file
    @type handle: file handle
    @ivar header: first lines of the file
    @type header: string
    @ivar started: whether some transcripts have already been writted
    @type started: boolean
    @ivar lastChromosome: the chromosome on which the transcript which was inserted last
    @type lastChromosome: string
    """


    def __init__(self, fileName, verbosity = 0):
        """
        Constructor
        @param fileName: name of the file 
        @type fileName: string
        @param verbosity: verbosity
        @type verbosity: int
        """
        self.fileName = fileName
        self.verbosity = verbosity
        self.handle = open(self.fileName, "w")
        self.started = False
        self.lastChromosome = None
        self.header = ""
        self.sequenceFileName = None


    def __del__(self):
        """
        Destructor
        """
        self.close()


    def close(self):
        """
        Close writer
        """
        if self.handle != None and not self.handle.closed:
            self.handle.close()
        self.handle = None


    def addTranscript(self, transcript):
        """
        Add a transcript to the list of transcripts to be written
        @param transcript: transcript to be written
        @type    transcript: class L{Transcript<Transcript>}
        """
        if not self.started:
            self.handle.write(self.header)
            self.started = True

        self.handle.write(self.printTranscript(transcript))
        self.lastChromosome = transcript.getChromosome()


    def addElement(self, element):
        """
        Same as "addTranscript"
        @param element: transcript to be written
        @type    element: class L{Transcript<Transcript>}
        """
        self.addTranscript(element)


    def addTranscriptList(self, transcriptList):
        """
        Add a list of transcripts to the transcripts to be written
        @param transcriptList: transcripts to be written
        @type    transcriptList: class L{TranscriptList<TranscriptList>}
        """
        progress = Progress(transcriptList.getNbTranscripts(), "Writing transcripts", self.verbosity)
        for transcript in transcriptList.getIterator():
            self.addTranscript(transcript)
            progress.inc()
        progress.done()
            

    def addTranscriptTable(self, transcriptTable):
        """
        Add a list of transcripts in a mySQL table to the transcripts to be written
        @param transcriptTable: transcripts to be written
        @type    transcriptTable: class L{MySqlTranscriptTable<MySqlTranscriptTable>}
        """
        for transcript in transcriptTable.getIterator():
            self.addTranscript(transcript)
            
            
    def setTitle(self, title):
        """
        Possibly write a title for the list (by default, do nothing)
        @param title: a title for the list
        @type title:    string
        """
        pass
    
    def setFeature(self, feature):
        """
        Set the name of the feature
        @param title: the title of the feature
        @type    feature: string
        """
        pass
        
    def setFeaturePart(self, featurePart):
        """
        Set the name of the feature part
        @param title: the title of the feature part
        @type    featurePart: string
        """
        pass


    def addSequenceFile(self, fileName):
        """
        Get the multi-fasta file of the sequences
        """
        self.sequenceFileName = fileName
        
        
    def write(self):
        """
        No-op
        """
        pass
