#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import os
import random
from commons.core.writer.TranscriptListWriter import TranscriptListWriter
from SMART.Java.Python.misc.Progress import Progress

class CsvWriter(TranscriptListWriter):
    """
    A class that writes a transcript list into a file with CSV (Excel) format
    @ivar fileName: name of the file 
    @type fileName: string
    @ivar handle: handle to the file
    @type handle: file handle
    """


    def __init__(self, fileName, verbosity = 0):
        """
        Constructor
        @param fileName: name of the file 
        @type fileName: string
        @param verbosity: verbosity
        @type verbosity: int
        """
        super(CsvWriter, self).__init__(fileName, verbosity)
        self.header   = ""
        self.title    = "chromosome,start,end,strand,exons,tags\n"
        self.modified = False
            

    def __del__(self):
        """
        Destructor
        (Trick to write 1 tag per column)
        """
        if self.handle != None:
            self.modifyCsv()
        super(CsvWriter, self).__del__()


    def close(self):
        if self.handle != None:
            self.modifyCsv()
        super(CsvWriter, self).close()


    def modifyCsv(self):
        """
        Clean CSV file so that there is one column per tag
        """
        if self.modified:
            return

        # read all the tags
        self.handle.close()
        self.handle = open(self.fileName)
        nbFirstFields = 5
        tags = set()
        if self.verbosity >= 10:
            print "Modifying CSV file..."
        number = -1
        for number, line in enumerate(self.handle):
            if number != 0:
                theseTags = line.strip().split(",")[nbFirstFields:]
                for tag in theseTags:
                    if tag.find("=") != -1:
                        (key, value) = tag.split("=", 1)
                        if value != None:
                            tags.add(key)
        if self.verbosity >= 10:
            print " ...done"

        # re-write the file
        tmpFileName = "tmpFile%d.csv" % (random.randint(0, 100000))
        tmpFile = open(tmpFileName, "w")
        self.handle.seek(0)
        progress = Progress(number + 1, "Re-writting CSV file", self.verbosity)
        tmpFile.write(self.title.replace("tags", ",".join(sorted(tags))))
        for line in self.handle:
            tagValues = dict([(key, None) for key in tags])
            tmpFile.write(",".join(line.strip().split(",")[:nbFirstFields]))
            for tag in line.strip().split(",")[nbFirstFields:]:
                if tag.find("=") != -1:
                    key = tag.split("=", 1)[0]
                    tagValues[key] = tag.split("=", 1)[1]
                else:
                    tagValues[key] += ";%s" % (tag)
            for key in sorted(tagValues.keys()):
                tmpFile.write(",%s" % (tagValues[key]))
            tmpFile.write("\n")
            progress.inc()
        tmpFile.close()

        # replace former file
        import shutil
        shutil.move(tmpFile.name, self.fileName)
        progress.done()
        self.modified = True


    @staticmethod
    def getFileFormats():
        """
        Get the format of the file
        """
        return ["csv", "xls", "excel"]


    @staticmethod
    def getExtension():
        """
        Get the usual extension for the file
        """
        return "csv"
        
        
    def printTranscript(self, transcript):
        """
        Export the given transcript with GFF2 format
        @param transcript: transcript to be printed
        @type transcript: class L{Transcript<Transcript>}
        @return: a string
        """
        return transcript.printCsv()

