#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import os
import sys
from commons.core.writer.WriterChooser import WriterChooser
from commons.core.writer.MySqlTranscriptWriter import MySqlTranscriptWriter

class TranscriptWriter(object):
    """
    An interface class that writes a list of transcripts, handle different formats
    @ivar container: container of the data
    @type container: L{TranscriptContainer<TranscriptContainer>}
    @ivar format: format of the data to be printed
    @type format: string     
    @ivar file: the file where to print
    @type file: string 
    @ivar type: type of the data (transcripts, mappings or mySQL)
    @type type: string
    @ivar writer: a transcript list writer
    @type writer: L{TranscriptListWriter<TranscriptListWriter>} or None
    @ivar mode: use a container or enter transcript one by one
    @type mode: string
    @ivar verbosity: verbosity
    @type verbosity: int        
    """

    def __init__(self, file, format, verbosity = 0):
        """
        Constructor
        @param container: container of the data
        @type container: string
        @param format: format of the data
        @type format: string
        @param file: file where to print
        @type file: string
        @param verbosity: verbosity
        @type verbosity: int
        """
        self.container = None
        self.format = format
        self.file = file 

        self.verbosity = verbosity
        self.type = None
        self.writer = None
        self.mode = None
        if self.format == None:
            sys.exit("Error! Writer input format is empty!")

        if self.format == "sql":
            self.type = "sql"
            pos = self.file.rfind(os.sep)
            if pos > -1:
                self.file = self.file[pos+1:]
            self.writer = MySqlTranscriptWriter(self.file, self.verbosity)
        else:
            writerChooser = WriterChooser(self.verbosity)
            writerChooser.findFormat(self.format)
            self.writer = writerChooser.getWriter(self.file)
            self.type = writerChooser.getType()
            
            
    def close(self):
        """
        Close writer
        """
        if self.writer != None:
            self.writer.close()


    def setContainer(self, container):
        """
        Set a container for the data
        @param container: container of the data
        @type container: class L{TranscriptContainer<TranscriptContainer>}
        """
        self.container = container
        if self.mode == "transcript":
            raise Exception("Error! TranscriptWriter '%s' on 'transcript' mode is currently used on 'container' mode." % (self.file))
        self.mode = "container"


    def addTranscript(self, transcript):
        """
        Add a transcript to write
        @param transcript: a transcript
        @type transcript: class L{Transcript<Transcript>}
        """
        self.writer.addTranscript(transcript)
        if self.mode == "container":
            sys.exit("Error! TranscriptWriter '%s' on 'container' mode is currently used on 'transcript' mode." % (self.file))
        self.mode = "transcript"
        
        
    def addElement(self, transcript):
        """
        Same as addTranscript
        """
        self.addTranscript(transcript)
    

    def setTitle(self, title):
        """
        Possibly write a title for the list
        @param title: a title for the list
        @type title: string
        """
        if self.writer != None:
            self.writer.setTitle(title)

    def setFeature(self, feature):
        """
        Possibly Set the name of the feature
        @param title: the title of the feature
        @type    feature: string
        """
        if self.writer != None:
            self.writer.setFeature(feature)
        
    def setFeaturePart(self, featurePart):
        """
        Possibly Set the name of the feature part
        @param title: the title of the feature part
        @type    featurePart: string
        """
        if self.writer != None:
            self.writer.setFeaturePart(featurePart)    
        
    def setStrands(self, strands):
        """
        Possibly consider both strands separately
        @param strands: whether both strands should be considered separately
        @type  strands: boolean
        """
        if self.writer != None:
            self.writer.setStrands(strands)
            
        
    def write(self):
        """
        Write the content and possibly convert data
        """        
        if self.type == "transcript" or self.type == "sequence":
            if self.mode == "container":
                self.writer.addTranscriptList(self.container)
            return

        if self.mode == "transcript" or self.type == "sequence":
            self.writer.write()
            return

        if self.container.format != "sql":
            self.container.storeIntoDatabase()
        tables = self.container.getTables()
        for chromosome in tables:
            tables[chromosome].rename("%s_%s" % (self.file, chromosome))
        return
        

    def addSequenceFile(self, fileName):
        self.writer.addSequenceFile(fileName)
            