#
# Copyright INRA-URGI 2009-2010
# 
# This software is governed by the CeCILL license under French law and
# abiding by the rules of distribution of free software. You can use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
# 
# As a counterpart to the access to the source code and rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty and the software's author, the holder of the
# economic rights, and the successive licensors have only limited
# liability.
# 
# In this respect, the user's attention is drawn to the risks associated
# with loading, using, modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean that it is complicated to manipulate, and that also
# therefore means that it is reserved for developers and experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and, more generally, to use and operate it in the
# same conditions as regards security.
# 
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.
#
import sys
from commons.core.writer.TranscriptListWriter import TranscriptListWriter
from commons.core.writer.SequenceListWriter import SequenceListWriter
from commons.core.writer.BedWriter import BedWriter
from commons.core.writer.CsvWriter import CsvWriter
from commons.core.writer.EmblWriter import EmblWriter
from commons.core.writer.FastaWriter import FastaWriter
from commons.core.writer.FastqWriter import FastqWriter
from commons.core.writer.GbWriter import GbWriter
from commons.core.writer.Gff2Writer import Gff2Writer
from commons.core.writer.SamWriter import SamWriter
from commons.core.writer.UcscWriter import UcscWriter
from commons.core.writer.WigWriter import WigWriter
from commons.core.writer.Gff3Writer import Gff3Writer
from commons.core.writer.GtfWriter import GtfWriter
from commons.core.writer.MapWriter import  MapWriter

 
class WriterChooser(object):
    """
    A class that finds the correct writer
    @ivar type: transcript / sequence writer
    @type type: string
    @ivar format: the format of the writer
    @type format: string
    @ivar writerClass: the class of the writer
    @type writerClass: string
    @ivar extension: default extension of the file
    @type extension: string
    @ivar verbosity: verbosity
    @type verbosity: int        
    """

    def __init__(self, verbosity = 0):
        """
        Constructor
        @param verbosity: verbosity
        @type    verbosity: int
        """
        self.type = None
        self.format = None
        self.writerClass = None
        self.extension = None
        self.verbosity = verbosity
    

    def findFormat(self, format, type = None):
        """
        Find the correct parser
        @ivar format: the format
        @type format: string
        @ivar type: transcript sequence parser (None is all)
        @type type: string
        @return: a parser
        """
        classes = {}
        if (type == "transcript"):
            classes = {TranscriptListWriter: "transcript"}
        elif (type == "sequence"):
            classes = {SequenceListWriter: "sequence"}
        elif (type == None):
            classes = {TranscriptListWriter: "transcript", SequenceListWriter: "sequence"}
        else:
            sys.exit("Do not understand format type '%s'" % (type))

        for classType in classes:
            for writerClass in classType.__subclasses__():
                if format in writerClass.getFileFormats():
                    self.writerClass = writerClass
                    self.extension = writerClass.getExtension()
                    self.type = classes[classType]
                    return
        sys.exit("Cannot get writer for format '%s'" % (format))


    def getWriter(self, fileName):
        """
        Get the writer previously found
        @return: the writer
        """
        return self.writerClass(fileName, self.verbosity)


    def getType(self):
        """
        Get the type of writer previously found
        @return: the type of writer
        """
        return self.type


    def getExtension(self):
        """
        Get the default extension of writer previously found
        @return: the extension
        """
        return self.extension

